function [rho, probeResponse] = getSteadySolution(RAQR_config, steadyEfield)
% This function returns the zero-temperature steady state density matrix 

    % Physical constants
    epsilon_0   = 8.85e-12; 
    mu_0        = 4*pi*1e-7; 
    hbar        = 6.626e-34 / (2*pi); 
    c0          = physconst('LightSpeed'); 

    a0      = 5.2918e-11; 
    e       = 1.6e-19; 
    mu_MW   = 1443.45*e*a0; 
    mu_12   = 4.5022*e*a0;          

    % Extract parameters 

    d           = RAQR_config.d; 
    N0          = RAQR_config.N0; 
    lambda_p    = RAQR_config.lambda_p; 

    gamma   = RAQR_config.gamma; 
    gamma_2 = RAQR_config.gamma_2; 
    gamma_3 = RAQR_config.gamma_3; 
    gamma_4 = RAQR_config.gamma_4; 
    gamma_c = RAQR_config.gamma_c; 
    
    Omega_c = RAQR_config.Omega_c; 
    Omega_p = RAQR_config.Omega_p; 

    Delta_c = RAQR_config.Delta_c; 
    Delta_p = RAQR_config.Delta_p; 
    Delta_l = RAQR_config.Delta_l; 

    Omega_RF = (steadyEfield*mu_MW)/(hbar); 
    Delta_p = -Delta_p; Delta_c = -Delta_c; Delta_l = -Delta_l;     % Negate all the detuning values.  

    H = [0,         Omega_p/2,  0,                      0;                  ...
        Omega_p/2,  Delta_p,    Omega_c/2,              0;                  ...
        0,          Omega_c/2,  Delta_p+Delta_c,        Omega_RF/2;         ...
        0,          0,          conj(Omega_RF)/2,       Delta_p+Delta_c+Delta_l]; 
    
    Gamma = diag([gamma, gamma+gamma_2, gamma+gamma_3+gamma_c, gamma+gamma_4]); 


    % fc = 6.9458e9;  % Transition frequency  

    % Re-order the elements of ρ to construct a vector differential equation 
    % dx/dt = Ax + b
    Aorder = [  1, 1; ...
                2, 2; ...
                3, 3; ...
                4, 4; ...
                1, 2; ...
                1, 3; ...
                1, 4; ...
                2, 3; ...
                2, 4; ...
                3, 4; ...
                2, 1; ...
                3, 1; ...
                3, 2; ...
                4, 1; ...
                4, 2; ...
                4, 3; ];  
    
    [Nvar, ~]   = size(Aorder); 
    A           = zeros(Nvar); 
    % b           = zeros(Nvar, 1); 
    % b(1)        = gamma;    
    
    for ii = 1:Nvar
        for jj = 1:Nvar
            m = Aorder(ii, 1); 
            n = Aorder(ii, 2); 
            k = Aorder(jj, 1); 
            l = Aorder(jj, 2); 
            A(ii, jj) = getA(m, n, k, l, H, Gamma, gamma, gamma_2, gamma_3, gamma_4); 
        end
    end
    
    % Try to find the steady-state solution for Ax=0
    largeNumber = 1e4; 
    Aext        = [A; largeNumber*[ones(1, 4), zeros(1, 12)]];      % 17x16 of rank 16
    [U, S, V]   = svd(Aext); 
    s           = diag(S); 
    tmp         = (U'*[zeros(16, 1); largeNumber]); 
    xvec_steady = V*inv(diag(s))*tmp(1:16); 
    
    rho = zeros(4); 
    for ii = 1:Nvar
        m = Aorder(ii, 1); 
        n = Aorder(ii, 2); 
        rho(m, n) = xvec_steady(ii); 
    end
    
    % compute probe transmission [dB] 
    probeResponse = (20/log(10))*(-pi*d/lambda_p)*imag(-(2*N0*mu_12^2)/(epsilon_0*hbar*Omega_p)*rho(2,1)); 
end

%% Functional utils 
function A = getA(m, n, k, l, H, Gamma, gamma, gamma_2, gamma_3, gamma_4)
    A = 0; 

    % -i (Hρ) term
    A = A + (-1i)*(n==l)*H(m,k); 
    
    % i(ρH) term
    A = A + (1i)*(m==k)*H(l,n); 

    % -(1/2)Gamma*ρ term
    A = A - (1/2)*(n==l)*Gamma(m,k); 

    % -(1/2)ρ*Gamma term
    A = A - (1/2)*(m==k)*Gamma(l,n); 

    if k == 2 && l == 2 && m == 1 && n == 1
        A = A + gamma_2;
    end

    if k == 4 && l == 4 && m == 1 && n == 1
        A = A + gamma_4; 
    end

    if k == 3 && l == 3 && m == 2 && n == 2
        A = A + gamma_3; 
    end

end

