% ======================================================= % 
% 
%  This file studies the multi-band quantum receiver, and 
%  plots the two quantum transconductances w.r.t. IF freq. 
% 
% ======================================================= % 


clear; clc; close all; 

RAQR_config = configureRAQR("DualBand"); 

% Re-configure some key parameters for tuning the system 
RAQR_config.A_LO1       = 0.004; 
RAQR_config.A_LO2       = 0.003;            
% These parameters will greatly influence the gain factor \kappa, as is observed in EIT-AT simulations. 
% Thus, if they are not considered, the DC transfer gains will significantly deviate from the true values. 
% RAQR_config.gamma_4     = 2*pi*0.01e3;   
% RAQR_config.gamma_3     = 2*pi*0.12e3; 
% RAQR_config.gamma_5     = 2*pi*1e2; 


% Some useful physical constants. 
hbar        = 6.626e-34 / (2*pi); 
a0          = 5.2918e-11; 
e           = 1.6e-19; 
mu_RF1      = 1443.45*e*a0; 
mu_RF2      = mu_RF1; 
epsilon_0   = 8.85e-12;
mu_12       = 4.5022*e*a0; 
c0          = physconst('LightSpeed');
N0          = RAQR_config.N0;
kp          = 2*pi/(RAQR_config.lambda_p); 


Delta_p     = RAQR_config.Delta_p; 
Delta_c     = RAQR_config.Delta_c; 
Delta_l1    = RAQR_config.Delta_l1; 
Delta_l2    = RAQR_config.Delta_l2; 

gamma       = RAQR_config.gamma; 
gamma_2     = RAQR_config.gamma_2; 
gamma_3     = RAQR_config.gamma_3; 
gamma_4     = RAQR_config.gamma_4; 
gamma_5     = RAQR_config.gamma_5; 
gamma_c     = RAQR_config.gamma_c; 

Omega_c     = RAQR_config.Omega_c; 
Omega_p     = RAQR_config.Omega_p;  

Omega_RF1   = mu_RF1*(RAQR_config.A_LO1)/hbar;    
Omega_RF2   = mu_RF2*(RAQR_config.A_LO2)/hbar; 


% Quantum state definition 
%  |1> 6S1/2    |2> 6P3/2   |3> 47D5/2          |4> 48P3/2                  |5>: Some other Rydberg state 

H = [0,         Omega_p/2,  0,                  0,                          0;                                  ...
    Omega_p/2,  -Delta_p,    Omega_c/2,         0,                          0;                                  ...
    0,          Omega_c/2,  -Delta_p-Delta_c,   Omega_RF1/2,                Omega_RF2/2;                        ...
    0,          0,          Omega_RF1/2,        -Delta_p-Delta_c+Delta_l1,   0;                                  ... 
    0,          0,          Omega_RF2/2,        0,                          -Delta_p-Delta_c+Delta_l1-Delta_l2]; 


Gamma   = diag([gamma, gamma+gamma_2, gamma+gamma_3, gamma+gamma_4, gamma+gamma_5]); 

didx    = zeros(1, 5); 
for idx = 1:5
    didx(idx) = (idx-1)*5+idx; 
end

A0                      = -1i*(kron(eye(5), H) - kron(H.', eye(5))) - (1/2)*(kron(eye(5), Gamma) + kron(Gamma, eye(5))); 
A0(didx(1), didx(2))    = A0(didx(1), didx(2)) + gamma_2; 
A0(didx(1), didx(4))    = A0(didx(1), didx(4)) + gamma_4; 
A0(didx(2), didx(3))    = A0(didx(2), didx(3)) + gamma_3;
A0(didx(1), didx(5))    = A0(didx(1), didx(5)) + gamma_5; 

u5          = zeros(1, 16); 
u5(didx)    = 1/5;

eps = 1e-6; 
assert(norm(u5*A0) <= eps, "A0 matrix error: Trace-preserving property not satisfied.\n");

Qtmp        = eye(5^2); 
Qtmp(:,1)   = (u5.')/norm(u5); 
[Q, ~]      = qr(Qtmp); 
Q(:,1)      = -Q(:, 1); 

B0      = Q'*A0*Q; 
C0      = B0(2:end, 2:end); 
w0      = B0(2:end, 1); 
assert(norm(B0(1,:)) <= eps); 

y_1bar  = 1/sqrt(5);  
z0bar   = -C0\(y_1bar*w0); 
x0bar   = Q*[y_1bar; z0bar];            % Steady-state x solution 
rho_bar = reshape(x0bar, [5,5]);        % Steady-state density matrix solution 

% Show the steady-state solution 
% figure(2); 
% imagesc(abs(rho_bar)); colorbar; 

alpha_0             = -(kp*N0*(mu_12^2))/(epsilon_0*hbar*Omega_p)*imag(rho_bar(2,1)); 
P_probe             = 29.8e-6; 
probeTransmission   = exp(-2*alpha_0*RAQR_config.d); 
Iph0                = (RAQR_config.eta*P_probe)/(hbar*(c0*kp))*e*probeTransmission; 
gq_PreFactor        = Iph0*(2*kp*N0*(mu_12^2))/(epsilon_0*hbar*Omega_p); 


H43     = zeros(5); H43(4,3) = 1; 
E43     = -1i*(kron(eye(5), H43) - kron(H43.', eye(5)));
E34     = -1i*(kron(eye(5), H43.') - kron(H43, eye(5)));
F43     = Q'*E43*Q; 
F34     = Q'*E34*Q; 

H53     = zeros(5); H53(5,3) = 1; 
E53     = -1i*(kron(eye(5), H53) - kron(H53.', eye(5)));
E35     = -1i*(kron(eye(5), H53.') - kron(H53, eye(5)));
F53     = Q'*E53*Q; 
F35     = Q'*E35*Q; 

% Another method for evaluating the quantum transconductances. 
mu_arr    = [mu_RF1; mu_RF2]; 
gamma_arr = [gamma_4; gamma_5]; 
delta_arr = [-Delta_l1; Delta_l2]; 
A_LO_arr  = [RAQR_config.A_LO1; RAQR_config.A_LO2]; 
[Tq, dTq, rho_bar_new] = getDCTransferCoeffs(RAQR_config, 2, mu_arr, gamma_arr, delta_arr, A_LO_arr); 

gqs_new   = gq_PreFactor*mu_arr/(2*hbar).*imag(Tq);
dgq_new   = gq_PreFactor*(mu_arr/(2*hbar)).*imag(dTq).*(mu_arr.'/(2*hbar)); 
mag2db(abs(gqs_new)*-1e3)

f_arr       = logspace(2, 8, 2e3).'; 
gq1_arr     = zeros(size(f_arr)); 
gq2_arr     = zeros(size(f_arr)); 

% ZJA: we may evaluate the Jacobian of d[g_q1, g_q2]/d[A_LO1, A_LO2]. 
% There are 4 values to evaluate. 
% Then, something could be optimized. 
% ZJA: I have evaluated this thing now. 

for idx = 1:length(f_arr)
    svar        = 1i*2*pi*f_arr(idx); 

    T34         = Q*[zeros(1, 24); eye(24)]* inv(svar*eye(24)-C0) * F34(2:end,2:end) * z0bar; 
    T34         = T34(2);
    T43         = Q*[zeros(1, 24); eye(24)]* inv(svar*eye(24)-C0) * F43(2:end,2:end) * z0bar; 
    T43         = T43(2);
    
    T34_nf      = Q*[zeros(1, 24); eye(24)]* inv(-svar*eye(24)-C0) * F34(2:end,2:end) * z0bar; 
    T34_nf      = T34_nf(2);
    T43_nf      = Q*[zeros(1, 24); eye(24)]* inv(-svar*eye(24)-C0) * F43(2:end,2:end) * z0bar; 
    T43_nf      = T43_nf(2); 

    gq1_arr(idx)= gq_PreFactor * (mu_RF1/(2*hbar))*((T34-conj(T34_nf))/(2i)+(T43-conj(T43_nf))/(2i)); 

    T35         = Q*[zeros(1, 24); eye(24)]* inv(svar*eye(24)-C0) * F35(2:end,2:end) * z0bar; 
    T35         = T35(2);
    T53         = Q*[zeros(1, 24); eye(24)]* inv(svar*eye(24)-C0) * F53(2:end,2:end) * z0bar; 
    T53         = T53(2);
    
    T35_nf      = Q*[zeros(1, 24); eye(24)]* inv(-svar*eye(24)-C0) * F35(2:end,2:end) * z0bar; 
    T35_nf      = T35_nf(2);
    T53_nf      = Q*[zeros(1, 24); eye(24)]* inv(-svar*eye(24)-C0) * F53(2:end,2:end) * z0bar; 
    T53_nf      = T53_nf(2); 

    gq2_arr(idx)= gq_PreFactor * (mu_RF2/(2*hbar))*((T35-conj(T35_nf))/(2i)+(T53-conj(T53_nf))/(2i)); 

end


%% Visualizations 
% close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');

figure(1); 
yyaxis left; 
plot(f_arr, mag2db(abs(gq1_arr)*1e3)); 
set(gca, 'xscale', 'log'); grid on; hold on; 
plot(f_arr, mag2db(abs(gq2_arr)*1e3)); 
set(gca, 'xlim', [f_arr(1), f_arr(end)]); 
xlabel('Freq (Hz)'); 
ylabel('g_q (dBmS)'); 

yyaxis right; 
plot(f_arr, rad2deg(unwrap(angle(gq1_arr)))); grid on; hold on; 
plot(f_arr, rad2deg(unwrap(angle(gq2_arr))));
ylabel('Phase (deg)'); 

legend({'Band-1', 'Band-2'}, 'Location', 'best'); 

%% Utils 

