% ================================================= % 
% 
%  This script is meant for the study of atomic response
%  considering the Doppler broadening effect and the 
%  transit effect.
% 
%  Author: Jieao Zhu  Date: 11 Oct 2025 
% 
% ================================================= % 

clear; close all; clc; 


%%  Setup quantum system parameters 
RAQR_config = configureRAQR("Transit"); 

% Re-configure some important parameters 
% RAQR_config.Omega_c     = 2*pi*2.05e6;             % coupling beam 17.0mW  @ 510nm
% RAQR_config.Omega_p     = 2*pi*8.08e6;             % probe    beam 29.8uW  @ 852nm

% PARAMETERs (Omega_c, Omega_p) = (0.97M, 5.7M); (2.05M, 8.08M); (8M, 2M) 
% RAQR_config.A_LO = 0.4; 

eps = 1e-6; 

freq_arr = logspace(1, 8, 150); 
A_LO_arr = logspace(-1, 1, 50); 
Nf      = length(freq_arr); 
N_ALO   = length(A_LO_arr); 

gqMat   = zeros(N_ALO, Nf); 
ptArr   = zeros(N_ALO, 2); 

gqMat_a  =zeros(N_ALO, Nf); 
ptArr_a = zeros(N_ALO, 1); 

for idx = 1:N_ALO 
    new_config      = RAQR_config; 
    new_config.A_LO = A_LO_arr(idx); 

    gq_struct       = getNumericalFreqResponse(new_config, new_config.Temperature, freq_arr); 
    gqMat(idx, :)   = gq_struct.gq; 
    ptArr(idx, 1)   = gq_struct.pt; 
    ptArr(idx, 2)   = gq_struct.pt_ztemp; 
    
    gq_struct_a     = getAnalyticalFreqResponse(new_config, new_config.Temperature, freq_arr); 
    gqMat_a(idx, :) = gq_struct_a.gq; 
    ptArr_a(idx, :) = gq_struct_a.pt; 

    fprintf('%d/%d\n', idx, N_ALO); 
end



%% Visualization 
close all; clc;
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',    12);
set(0,'DefaultAxesFontSize',    14);
set(0,'DefaultLineLineWidth',   1.5);
set(0,'defaultfigurecolor',     'w');

savefigs = false; 

% figure(1); 
% plot(freq_arr, mag2db(1e3*abs(gq_struct.gq_ztemp))); hold on; 
% plot(freq_arr, mag2db(1e3*abs(gq_struct.gq))); 
% 
% set(gca, 'xscale', 'log'); 
% grid on; 
% xlabel('IF Freq (Hz)'); 
% ylabel('g_q (dBmS)'); 
% 
% legend({"T=0K", "T=300K"}); 

[X, Y] = meshgrid(freq_arr, A_LO_arr/1e2);  

fig1 = figure(1);
fig1.Position = [413, 247, 704, 516]; 
subplot(2, 1, 1); 
surf(X, Y, mag2db(1e3*abs(gqMat))); colorbar; 
set(gca, 'xscale', 'log');  
xlabel('IF Freq (Hz)'); 
ylabel('LO Amplitude (Vp/cm)');
zlabel('g_q (dBmS) Numerical'); 
colormap('jet')
set(gca, 'yscale', 'log');
set(gca, 'xlim', [min(freq_arr), max(freq_arr)]); 
set(gca, 'ylim', [min(A_LO_arr), max(A_LO_arr)]/1e2); 
shading interp; 
view(0, 90); 
title('Numerical Freq Response'); 

subplot(2, 1, 2); 
surf(X, Y, mag2db(1e3*abs(gqMat_a))); colorbar; 
set(gca, 'xscale', 'log');  
xlabel('IF Freq (Hz)'); 
ylabel('LO Amplitude (Vp/cm)');
zlabel('g_q (dBmS) Analytical'); 
set(gca, 'yscale', 'log');
set(gca, 'xlim', [min(freq_arr), max(freq_arr)]); 
set(gca, 'ylim', [min(A_LO_arr), max(A_LO_arr)]/1e2); 
shading interp; 
view(0, 90); 
title('Analytical Freq Response'); 
if savefigs
    exportgraphics(fig1, "./results/response/T_NumericalAnalyticalFreqResponse.pdf", "ContentType", "vector"); 
    fprintf('Figure 1 saved.\n'); 
end


fig2 = figure(2); 
fig2.Position = [1137,311.4,560,420.00]; 
plot(A_LO_arr.'/1e2, ptArr_a, 'LineWidth', 2.0, 'LineStyle', '--'); grid on; hold on; 
plot(A_LO_arr.'/1e2, ptArr(:,1), 'LineWidth', 0.5, 'Marker', 'x'); 
plot(A_LO_arr.'/1e2, ptArr(:,2), 'LineWidth', 0.5, 'Marker', 'x');
set(gca, 'xscale', 'log'); 
xlabel('LO Amplitude (Vp/cm)'); 
ylabel('Probe transmission'); 
legend({"T=300K, Analytical", "T=300K, Numerical", "T=0K, Numerical"}); 

if savefigs
    exportgraphics(fig2, "./results/jpgs/T_NumericalAnalyticalPTResponse.jpg"); 
    % PT: Probe Transmission. 
    fprintf('Figure 2 saved.\n'); 
end
% TODO: We may find the transfer function from gq_arr. 
% We need some algorithm to do this. 
% We have a problem here. Why does pt increase as E_LO increase from 0? 


fprintf('End of this script.\n'); 
