% ================================================== % 
% 
%   This script generates the noise factor of each stage 
%   of RAQR. 
%   
%   RIN: Relative intensity noise 
% 
% ================================================== %

clear; close all; clc; 


% Setup quantum system parameters 
RAQR_config = configureRAQR(); 


kB      = getPhysicalConstant("Boltzmann"); 
T       = 300;                  % environment temperature 
fc      = 6.9458e9;   
e       = 1.6e-19; 
hbar    = 6.626e-34 / (2*pi); 
n0      = pow2db(kB*T)+30; 
zeta    = db2pow(-2); 


[gI1, gQ1, gI2, gQ2] = getQuantumTransConductanceTFs(RAQR_config, "tfs2.mat"); 

f_arr   = logspace(2, 8, 500); 
gI2_arr = freqs(gI2.b, gI2.a, 2*pi*f_arr); 
gq      = abs(gI2_arr(1)); 

c0      = getPhysicalConstant("LightSpeed"); 
eta0    = getPhysicalConstant("VacuumWaveImpedance"); 
bbn_psd = 4*pi/3*eta0*(2*(fc^2/c0^2)*kB*T)*zeta*(RAQR_config.d^2)*abs(gI2_arr).^2;  % BBR noise, PD output current PSD [A^2/Hz] 

Iph0        = getTransmittedProbePower0(RAQR_config); 
shot_psd    = e*Iph0*ones(size(f_arr));                                             % Photon shot noise, PD output current PSD [A^2/Hz]


% Noise equivalent power, optical IF-domain 
% C_pwr2cur   = RAQR_config.eta*e/(hbar*2*pi*(c0/RAQR_config.lambda_p)); 
% NEP_bb      = bbn_psd/(C_pwr2cur^2); 
% NEP_pd      = pd_psd/(C_pwr2cur^2); 
% NEP_shot    = shot_psd/(C_pwr2cur^2);


% Trans-impedance amplifier PARAMETERS (TIA model: DHPCA-100 Variable Gain High-speed Current Amplifier)
TIA_RT      = 10e3; 
TIA_Ipsd    = (1.8e-12/sqrt(2))^2;      % TIA input referred current noise PSD [A^2/Hz]
TIA_Vpsd    = (2.8e-9/sqrt(2))^2;       % TIA input referred voltage noise PSD [V^2/Hz] 
TIA_Zin     = 60; 
TIA_Zout    = 50;       
PD_Rs       = 1e3;                      % DC bias resistor of the PD 

currDiv_coef    = (PD_Rs)/(PD_Rs+TIA_Zin); % Currents are distributed according to their conductances. 
arrPSDw_bb      = bbn_psd*((currDiv_coef*TIA_RT)^2)/TIA_Zout;               % TIA output blackbody noise power PSD  in [W/Hz]    

% Johnson-Nyquist noise + TIA noise 
PSDout_Rs       = (2*kB*T/PD_Rs)*(TIA_RT^2)/TIA_Zout; 
PSDout_inTIA    = TIA_Ipsd*((currDiv_coef*TIA_RT)^2)/TIA_Zout;  
PSDout_vnTIA    = TIA_Vpsd*(TIA_RT/(PD_Rs+TIA_Zin))^2/TIA_Zout; 
arrPSDw_th      = (PSDout_Rs+PSDout_inTIA+PSDout_vnTIA)*ones(size(f_arr));  % TIA output thermal noise power PSD    in [W/Hz]
arrPSDw_shot    = shot_psd*((currDiv_coef*TIA_RT)^2)/TIA_Zout;              % TIA output photon shot noise PSD      in [W/Hz]


%% Noise factor analysis for each stage 

Rs_arr = logspace(2, 5, 100).'; 
Nscan = length(Rs_arr); 

F1_dBarr = zeros(Nscan, 1); 
G1_dBarr = zeros(Nscan, 1); 
F2_dBarr = zeros(Nscan, 1); 
G2_dBarr = zeros(Nscan, 1); 
F_dBarr  = zeros(Nscan, 1); 
G_dBarr  = zeros(Nscan, 1); 

NoisePercentage = zeros(Nscan, 5); 

for idx_scan = 1:Nscan
    PD_Rs           = Rs_arr(idx_scan); 
    fprintf('Rs = %.2f kOhm\n', PD_Rs/1e3); 
    currDiv_coef    = (PD_Rs)/(PD_Rs+TIA_Zin);

    % Amplification stage1: From spatial E-field (times L) [V] to photocurrent
    % [A]. The gain is quantum transconductance gq [S]. The gain cannot be
    % defined to be the usual power gain GT. 
    
    % Noise factor F1 is defined as (SNRi/SNRo) 
    RIN     = db2pow(-140);         % nonlinear laser relative intensity noise (RIN), assumed to be -140dBc/Hz  
    fc      = 6.9458e9; 
    Aeq     = 3*(c0/fc)^2/(8*pi);   % The effective area of an equivalent dipole 
    
    % F1 formula: output-referred total current noise PSD / output-referred
    % thermal (bbr) current noise PSD. ZJA-comment: the input-referred F1 and
    % the output-referred F1 are the same for electronic systems. 
    Esig    = 1; % unit input E-field.  
    % F1      = (bbn_psd(1) + shot_psd(1) + (2*kB*T/PD_Rs) + (Iph0^2*(RIN)/2) )/(bbn_psd(1)/2);   % input in-band noise is 3dB smaller than output in-band noise, due to lack of mirror freq suppression.  
    % F1      = (bbn_psd(1) + shot_psd(1) + (2*kB*T/PD_Rs) + (Iph0^2*(RIN)/2) )/(2*kB*T/PD_Rs); % The noise pwr ratio with/without PD signal 
    SNRin   = (abs(Esig)^2/(2*eta0)*Aeq)/(kB*T);           % HERE Esig: V/m/sqrt(Hz)

    % Noise components at the output of STAGE 1
    tot_PSDqout = bbn_psd(1) + shot_psd(1) + (Iph0^2*(RIN)/2) + (2*kB*T/PD_Rs); 
    SNRout  = (RAQR_config.d*(gq)*Esig)^2/4/tot_PSDqout; 
    F1      = SNRin/SNRout; 
    F1_dB   = pow2db(F1); 
    
    
    tot_PSD = tot_PSDqout + TIA_Ipsd*currDiv_coef^2+TIA_Vpsd/(PD_Rs+TIA_Zin)^2; 

    NoisePercentage(idx_scan, 1) = 100*(bbn_psd(1))/tot_PSD; 
    NoisePercentage(idx_scan, 2) = 100*(2*kB*T/PD_Rs)/tot_PSD; 
    NoisePercentage(idx_scan, 3) = 100*(Iph0^2*(RIN)/2)/tot_PSD; 
    NoisePercentage(idx_scan, 4) = 100*(shot_psd(1))/tot_PSD; 
    NoisePercentage(idx_scan, 5) = 100*(TIA_Ipsd*currDiv_coef^2+TIA_Vpsd/(PD_Rs+TIA_Zin)^2)/tot_PSD; 
    
    fprintf('BBR fraction = %.2f%%\n',              NoisePercentage(idx_scan, 1));  
    fprintf('Rs noise fraction = %.2f%%\n',         NoisePercentage(idx_scan, 2)); 
    fprintf('Laser RIN fraction = %.2f%%\n',        NoisePercentage(idx_scan, 3)); 
    fprintf('Laser shot noise fraction = %.2f%%\n', NoisePercentage(idx_scan, 4)); 
    fprintf('TIA noise fraction = %.2f%%\n',        NoisePercentage(idx_scan, 5)); 
    % Power Gain G1 
    
    % ZJA comment: There are many definitions of the power gains, e.g.,
    % available power gain, and operational power gain. In this code, we adopt
    % the operational power gain to enable the application of Friis noise
    % formula. 
    G1      = ((RAQR_config.d*(gq)*Esig*currDiv_coef)^2*TIA_Zin/2) / (Esig^2/(2*eta0)*Aeq); % G1 := Output power of PD / Equiv. RF input power. 
    % ZJA comment: Here the adoption of Aeq is to validate the definition of F1. For a classical electronic antenna, the definition of Aeq converts the classical 
    % BBR noise into an electronic circuit noise with available noise DS-PSD
    % kB*T/2. In this way, we can treat the quantum stage (stage 1) as a
    % classical RF LNA with noise factor F1 and power gain G1. 
    G1_dB   = pow2db(G1); 
    
    fprintf('Stage 1: Noise factor = %.2f dB; Power gain = %.2f dB\n', F1_dB, G1_dB); 
    
    
    % Amplification stage2: From photocurrent [A] to TIA output voltage [V]. We
    % Note that the TIA input may not be matched. We need to compute the
    % mismatched noise factor. 
    % Gamma_in = (TIA_Zin-PD_Rs)/(TIA_Zin+PD_Rs);
    % F2      = 1 + ( TIA_Ipsd*currDiv_coef^2+TIA_Vpsd/(PD_Rs+TIA_Zin)^2 )/(2*kB*T/PD_Rs); % 1 + (input-referred TIA current noise)/(input-referred thermal current noise)
    F2      = 1+( TIA_Ipsd*currDiv_coef^2+TIA_Vpsd/(PD_Rs+TIA_Zin)^2 )/(tot_PSDqout); 
    F2_dB   = pow2db(F2); 
    G2      = ((TIA_RT)^2/TIA_Zout)/(TIA_Zin); 
    G2_dB   = pow2db(G2); 
    fprintf('Stage 2: Noise factor = %.2f dB; Power gain = %.2f dB\n', F2_dB, G2_dB); 
    
    % Total noise factor, using Friis noise formula 
    F = F1 + (F2-1)/G1; 
    G = G1*G2; 
    
    F_dB = pow2db(F); 
    G_dB = pow2db(G); 
    
    fprintf('Total noise factor = %.2f dB; Power gain = %.2f dB\n', F_dB, G_dB); 
    
    F1_dBarr(idx_scan) = F1_dB; 
    G1_dBarr(idx_scan) = G1_dB; 
    F2_dBarr(idx_scan) = F2_dB; 
    G2_dBarr(idx_scan) = G2_dB; 
    F_dBarr(idx_scan) = F_dB; 
    G_dBarr(idx_scan) = G_dB; 
    fprintf('=============================================================\n'); 
end


%% Visualization 
close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');
saveFigs = false; 

fig1 = figure(1); 
yyaxis left; 
plot(f_arr, 1e3*abs(gI2_arr)); 
set(gca, 'xscale', 'log'); 
set(gca, 'yscale', 'log'); 
grid on; 
xlabel('IF freq (Hz)'); 
ylabel('g_q (mS)'); 

yyaxis right; 
plot(f_arr, rad2deg(unwrap(angle(gI2_arr)))); 
set(gca, 'xscale', 'log'); grid on; 
ylabel('phase (deg)'); 

fig2 = figure(2); 
plot(f_arr, pow2db(arrPSDw_th*1e3), 'color', 'b'); hold on; grid on; 
plot(f_arr, pow2db(arrPSDw_bb*1e3), 'color', 'magenta'); 
plot(f_arr, pow2db(arrPSDw_shot*1e3), 'color', 'r'); 

xlabel('IF freq (Hz)'); 
ylabel('Equiv. noise PSD (dBm/Hz)'); 
legend({'Circuit thermal noise', 'Blackbody radiation noise', 'Photon shot noise'}, 'Location', 'southwest'); 
set(gca, 'xscale', 'log'); 

if saveFigs
   exportgraphics(fig2, "results/noise_psd.pdf", "ContentType", "vector");  
   fprintf('noise_psd.pdf exported.\n'); 
end

fig3 = figure(3); 
yyaxis left; 
plot(Rs_arr, F_dBarr ); grid on; hold on; 
plot(Rs_arr, F1_dBarr); 
plot(Rs_arr, F2_dBarr); 
set(gca, 'XScale', 'log'); 
xlabel('PD bias resistor (Ohm)'); 
ylabel('Noise factor (dB)'); 


yyaxis right; 
plot(Rs_arr, G_dBarr); hold on; 
plot(Rs_arr, G1_dBarr); 
plot(Rs_arr, G2_dBarr); 
ylabel('Signal gain (dB)'); 


legend({'F', 'F_q', 'F_T_I_A', 'G', 'G_q', 'G_T_I_A'}, 'NumColumns', 2, 'Location', 'north'); 

if saveFigs
    exportgraphics(fig3, "results/nf.pdf", "ContentType", "vector"); 
    fprintf('nf.pdf exported.\n'); 
end


fig4 = figure(4); 
plot(Rs_arr, NoisePercentage(:, 1)); hold on; 
plot(Rs_arr, NoisePercentage(:, 2)); 
plot(Rs_arr, NoisePercentage(:, 3)); 
plot(Rs_arr, NoisePercentage(:, 4));
plot(Rs_arr, NoisePercentage(:, 5));
legend({'BBR', 'Rs', 'RIN', 'Photon shot', 'Amp noise'}); 
xlabel('Rs (Ohm)'); 
set(gca, 'xscale', 'log'); 
ylabel('Fraction (%)'); 
