% ======================================================= % 
%   
%   This script studies the WMMSE-based optimization for  
%   multi-band quantum receiver. 
%   
%   Author:    Jieao Zhu
%   Date:      10 July 2025 
%   Comments:  qSDMA and qSFDMA achievable rate CDF  
% 
% ======================================================= % 

clear; clc; close all; rng(0);

M = 2;      % Number of bands 
RAQR_config = configureRAQR("DualBand"); 

% Re-configure some key parameters for tuning the system 
RAQR_config.A_LO1       = 0.04; 
RAQR_config.A_LO2       = 0.03;      
RAQR_config.gamma_5     = 2*pi*1.6e3;  % This value should be aligned with the ARC calculation. 

% Some useful physical constants. 
hbar        = 6.626e-34 / (2*pi); 
a0          = 5.2918e-11; 
e           = 1.6e-19; 
mu_RF1      = 1443.45*e*a0; 
mu_RF2      = 1214.6*e*a0; 
epsilon_0   = 8.85e-12;
mu_12       = 4.5022*e*a0; 
c0          = getPhysicalConstant('LightSpeed'); 
eta0        = getPhysicalConstant('VacuumWaveImpedance'); 
kB          = getPhysicalConstant("Boltzmann"); 
N0          = RAQR_config.N0;
kp          = 2*pi/(RAQR_config.lambda_p); 


mu_arr    = [mu_RF1; mu_RF2]; 
gamma_arr = [RAQR_config.gamma_4; RAQR_config.gamma_5]; 
delta_arr = [-RAQR_config.Delta_l1; RAQR_config.Delta_l2]; 
A_LO_arr  = [RAQR_config.A_LO1; RAQR_config.A_LO2]; 
fc_arr    = [6.9e9; 28e9]; 


%% Uplink Rydberg-MIMO Settings 
Nt = 4;             % Number of UE Tx antennas 
Nr = 5;             % Number of Rydberg-BS Rx antennas, usually larger than Nt. 
S = min([Nt, Nr]);  % Max number of available data streams
K  = 3;             % Number of UEs per band

% Setup TIA Electronic parameters 
BW          = 100e3; 
Vref        = 1e-3; 
TIA_RT      = 10e3; 
TIA_Zin     = 60; 
TIA_Zout    = 50;       
PD_Rs       = 1000;                      
T           = 300; 

Kc          = (PD_Rs)/(PD_Rs+TIA_Zin); 
zeta        = db2mag(-2); 

C_sig       = (1/2/Vref)*(TIA_RT*Kc)*(RAQR_config.d)*sqrt(8*pi*eta0)*(fc_arr/c0);
nw_thv      = (1/sqrt(2))*(1/Vref)*(2.8e-9)        *(TIA_RT/(TIA_Zin+PD_Rs)) ;                      % TIA voltage noise
nw_thi      = (1/sqrt(2))*(1/Vref)*(1.8e-12)       *(Kc*TIA_RT);                                    % TIA current noise 
nw_thRs     = (1/sqrt(2))*(1/Vref)*sqrt(4*kB*T/PD_Rs)*(Kc*TIA_RT); 
sigma2_e    = BW*(nw_thv^2 + nw_thi^2 + nw_thRs^2); 

Cqms  = cell(M, 1); % Normalized BBR noise covariance matrix  
Cqms_IFdiv = cell(M, 1); 
for mm = 1:M
    alpha_tmp       = exp(-10-rand(1));
    Cqms{mm}        = (4/3)*kB*T*BW*zeta*toeplitz(alpha_tmp.^(0:Nr-1)); 
    Cqms_IFdiv{mm}  = Cqms{mm}/M; 
end

mcMats  = cell(M, 1);
for mm = 1:M
    mcMat{mm}   = getMutualCouplingMatrix(fc_arr(mm), Nr); 
end


%% Simulation Loops 
% Nscan       = 9;
% Pmax_arr    = 1e-3*logspace(-1,1,Nscan).'; 
Pmax_arr    = 1e-3*db2pow([5, 10]); 
Nscan       = length(Pmax_arr); 

Nmc         = 31e3; 
Np          = 31; 
Nmethod     = 6; 


SEArrs   = cell(Nscan, 1); 

alpha       = ones(M, K); 
vbs         = 0; 
method      = "Armijo-Goldstein"; 
NiterMax    = 150;


for idx_scan = 1:Nscan
    rng(2); 
    
    % Task dispatcher
    assert(mod(Nmc, Np) == 0); 
    NmcEach = Nmc/Np; 

    SEOfAllMethods = cell(Np, 1); 
    % This LOOP is PARFOR  
    parfor idx_p = 1:Np
        rng(idx_p+1); 

        SE_MCtrials  = zeros(NmcEach, Nmethod); 
        for idx_mc = 1:NmcEach
    
            % Generate max. power allocation matrix
            Pmax = zeros(M, K);  % in Watts
            for mm = 1:M
                for kk = 1:K
                    Pmax(mm, kk) = Pmax_arr(idx_scan)*(1+0.25*(1-2*rand(1))); 
                end
            end
            
            % Generate channel 
            Hmks    = cell(M, K);   % Uplink MU-MIMO Channels
            Hmks_mc = cell(M, K); 
            for mm = 1:M
                for kk = 1:K
                    rmk = 500 + 1000*rand(1);  % distance
                    Hmks{mm,kk} = (c0/fc_arr(mm))/(4*pi*rmk)*(randn([Nr, Nt])+1i*randn([Nr, Nt]))/sqrt(2); 
                    Hmks_mc{mm,kk} = mcMat{mm}*Hmks{mm,kk}; 
                end
            end
            eps         = 1e-4;     % Error tolerance
            
            % Method 1: qWMMSE q-opt enabled 
            lr_ALO      = 6e-6; 
            opt_out     = qWMMSE(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms, C_sig, alpha, Pmax, sigma2_e, NiterMax, eps, lr_ALO, method, vbs); 
            SE_MCtrials(idx_mc, 1) = opt_out.wRate_trace(end)/M; 
    
            % Method 2: qFDMA 
            opt_out     = qWMMSE_IFdiv(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms_IFdiv, C_sig, alpha, Pmax, sigma2_e/M, NiterMax, eps, lr_ALO, method, vbs); 
            SE_MCtrials(idx_mc, 2) = opt_out.wRate_trace(end)/M; 

            % Method 3: qSDMA without qOpt 
            lr_ALO      = 0;
            opt_out     = qWMMSE(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms, C_sig, alpha, Pmax, sigma2_e, NiterMax, eps, lr_ALO, method, vbs); 
            SE_MCtrials(idx_mc, 3) = opt_out.wRate_trace(end)/M;     % final rate
    
            % % Method 4: qFDMA without qOpt
            lr_ALO      = 0; 
            opt_out     = qWMMSE_IFdiv(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms_IFdiv, C_sig, alpha, Pmax, sigma2_e/M, NiterMax, eps, lr_ALO, method, vbs); 
            SE_MCtrials(idx_mc, 4) = opt_out.wRate_trace(end)/M; 
            % 
            % Method 5: cSDMA without mutual coupling 
            opt_out = cWMMSE(Hmks, alpha, Pmax, BW*kB*T, NiterMax, eps, vbs); 
            SE_MCtrials(idx_mc, 5) = opt_out.wRate_trace(end)/M; 
            % 
            % % Method 6: cSDMA with mutual coupling 
            opt_out = cWMMSE(Hmks_mc, alpha, Pmax, BW*kB*T, NiterMax, eps, vbs); 
            SE_MCtrials(idx_mc, 6) = opt_out.wRate_trace(end)/M; 
            
        end

        SEOfAllMethods{idx_p} = SE_MCtrials; 
        fprintf('Scan %d/%d Pickle %d/%d complete. \n', idx_scan, Nscan, idx_p, Np); 
    end
    
    SEArr = zeros(Nmc, Nmethod); 
    for idx_p = 1:Np
        tmp = SEOfAllMethods{idx_p}; 
        SEArr((idx_p-1)*NmcEach+1:idx_p*NmcEach, :) = tmp; 
    end
    SEArrs{idx_scan} = SEArr;  

end

save('optHist.mat'); 
fprintf('Server: File saved at ./optHist.mat\n'); 

exit; 


%% Visualization 
load optHist.mat; 

close all; clc; 

set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor',    'w');
saveFigs = true; 

colors = interp1((0:255)/256*6, colormap('turbo'), flip(1/2+(0:(6-1))), "linear", "extrap" ); 

fig1 = figure(1); 
fig1.Position = [744,530,560,420]; 

% Sub-Fig1 with lower power 
idx_p = 1; 
rateData    = SEArrs{idx_p}; 
Rmin = floor(min([min(rateData, [], 'all'), min(rateData, [], 'all')])); 
Rmax = 25; 

Nbins = 30; 
subplot(2, 1, 1); 
histogram(rateData(:, 1), Nbins, 'FaceAlpha', 0.5, 'Normalization', 'probability'); hold on; grid on; 
histogram(rateData(:, 3), Nbins, 'FaceAlpha', 0.5, 'Normalization', 'probability');
histogram(rateData(:, 6), Nbins, 'FaceAlpha', 0.5, 'Normalization', 'probability');
xlim([Rmin, Rmax]); 
legend({'qSDMA-Opt', 'qSDMA-NoOpt', 'cSDMA-MC'}, 'FontSize', 10, 'Location', 'northeast'); 
title(sprintf('P_m_a_x=%.1f dBm', pow2db(1e3*Pmax_arr(idx_p))));  
ylabel('Probability'); 


% Sub-Fig2 with high power 
idx_p = 2; 
rateData    = SEArrs{idx_p}; 
subplot(2, 1, 2); 
histogram(rateData(:, 1), Nbins, 'FaceAlpha', 0.5, 'Normalization', 'probability'); hold on; grid on; 
histogram(rateData(:, 3), Nbins, 'FaceAlpha', 0.5, 'Normalization', 'probability'); 
histogram(rateData(:, 6), Nbins, 'FaceAlpha', 0.5, 'Normalization', 'probability'); 
xlim([Rmin, Rmax]); 
xlabel('Achievable spectral efficiency (bps/Hz)'); 
ylabel('Probability'); 
title(sprintf('P_m_a_x=%.1f dBm', pow2db(1e3*Pmax_arr(idx_p)))); 


if saveFigs
    exportgraphics(fig1, 'results/Technical_SE_Hist.pdf', 'ContentType','vector'); 
    fprintf('Technical_SE_Hist.pdf saved.\n'); 
end


