% ======================================================= % 
% 
%  This file studies the derivative of the quantum transconductance
% 
%  w.r.t. the change in probe/control Rabi frequency and LO amplitude A_LO.
%  
%  We may not optimize the probe Rabi frequency since it is related to too
%  many things, making this optimization unmeaningful.  
% 
% ======================================================= % 
clear; clc; close all; 

RAQR_config = configureRAQR(); 

hbar        = 6.626e-34 / (2*pi); 
a0          = 5.2918e-11; 
e           = 1.6e-19; 
mu_MW       = 1443.45*e*a0; 

RAQR_config.A_LO    = 0.04; 
% RAQR_config.Omega_c = 2*pi*3.9e6; 

% optimization loop 
first = true; 
loop = true; 
Nimax = 100; 

iter = 0; 

R_seq       = zeros(Nimax, 1); 
BW_seq      = zeros(Nimax, 1); 
gq_seq      = zeros(Nimax, 1); 
Omega_c_seq = zeros(Nimax, 1); 
A_LO_seq    = zeros(Nimax, 1); 
iter_seq    = (1:Nimax).'; 

while loop
    Delta_p     = RAQR_config.Delta_p; 
    Delta_c     = RAQR_config.Delta_c; 
    Delta_l     = RAQR_config.Delta_l; 
    
    
    gamma       = RAQR_config.gamma; 
    gamma_2     = RAQR_config.gamma_2; 
    gamma_3     = RAQR_config.gamma_3; 
    gamma_4     = RAQR_config.gamma_4; 
    gamma_c     = RAQR_config.gamma_c; 
    
    Omega_c     = RAQR_config.Omega_c; 
    Omega_p     = RAQR_config.Omega_p;  
    
    Omega_RF    = mu_MW*(RAQR_config.A_LO)/hbar;       
    
    % Quantum state definition 
    %  |1> 6S1/2    |2> 6P3/2   |3> 47D5/2          |4> 48P3/2 
    Delta_p = -Delta_p; Delta_c = -Delta_c; Delta_l = -Delta_l; 
    
    H = [0,         Omega_p/2,  0,                  0;                  ...
        Omega_p/2,  Delta_p,    Omega_c/2,          0;                  ...
        0,          Omega_c/2,  Delta_p+Delta_c,    Omega_RF/2;         ...
        0,          0,          Omega_RF/2,         Delta_p+Delta_c+Delta_l]; 
        
    Gamma       = diag([gamma, gamma+gamma_2, gamma+gamma_3+gamma_c, gamma+gamma_4]); 
    
        
    % Re-order the elements of ρ to construct a vector differential equation 
    % dx/dt = Ax + b
    
    A0          = -1i*(kron(eye(4), H) - kron(H.', eye(4))) - (1/2)*(kron(eye(4), Gamma) + kron(Gamma, eye(4))); 
    A0(1, 6)    = A0(1, 6) + gamma_2; 
    A0(1, 16)   = A0(1, 16) + gamma_4; 
    A0(6, 11)   = A0(6, 11) + gamma_3; 
    
    % Get steady-state solution 
    u4 = zeros(1, 16); u4([1, 6, 11, 16]) = 1; 
    
    Aext        = [A0; 1e4*u4];  
    [U, S, V]   = svd(Aext); 
    s           = diag(S); 
    tmp         = (U'*[zeros(16, 1); 1e4]); 
    xvec_steady = V*inv(diag(s))*tmp(1:16); 
    % 
    % fprintf('Steady-state rho2_1='); 
    % disp(xvec_steady(2)); 
    
    % Perform symmetrization on xvec_steady. This step is not always necessary.
    rho_steady  = reshape(xvec_steady, [4,4]); 
    rho_steady  = (rho_steady+rho_steady')/2; 
    xvec_steady = rho_steady(:); 
    
    
    Qtmp        = eye(16); 
    Qtmp(:,1)   = (u4.')/norm(u4); 
    [Q, ~]      = qr(Qtmp); 
    Q(:,1)      = -Q(:, 1); 
    
    % (y = Q'*x) is the variable substituion 
    B0          = Q'*A0*Q; 
    assert(norm(B0(1, :)) < 1e-6); % The first row of A is all-zero. 
    
    % Eqn: dy(2:n)/dt = y(1)*w + A_tilde*y(2:n), where y1(t) is always
    % constant = (1/2).
    w0      = B0(2:end, 1); 
    C0      = B0(2:end, 2:end); 
    
    % [Vtilde, Dtilde] = eig(A_tilde); 
    
    % Let y(2:16) = z
    z0bar   = -C0\((1/2)*w0); 
    
    % Recover x_steady from z_steady 
    y_steady = [(1/2); z0bar]; 
    x_steady = Q*y_steady; 
    
    
    % syms s; 
    % tmp = s*eye(15)-A_tilde; 
    % pA2 = sym2poly(det(tmp));  
    
    % Evaluate derivative of (gq) w.r.t. real-valued frequency f. 
    f       = 1e3; 
    svar    = 1i*2*pi*f; 
    eps     = 1e-6; 
    
    [E34, E43] = getActionMatrices(); % E34 and E43 are d(A1)/d(H1_{34,43}). 
    F34     = Q'*E34*Q; 
    F43     = Q'*E43*Q; 
    assert(norm(F34(2:16, 1))<=eps && norm(F43(2:16, 1))<=eps );   
    
   
    gq_PreFactor = getTransmittedProbePower0(RAQR_config, "gq_PreFactor");  
    
    
    
    % Use the conjugate technique to extract the imaginary transfer functions 
    % [gq, ~] = get_gq(mu_MW, hbar, Q, C0, z0bar, gq_PreFactor, F34, F43, f); 
   
    

    %% Construct some communication-oriented objective function 
    
    % Numerically find the bandwidth of the quantum system. This step is
    % somehow corrupted by the "chaotic response region". 
    f_arr       = logspace(3, 8, 4e3); 
    [gq_arr, ~] = get_gq(mu_MW, hbar, Q, C0, z0bar, gq_PreFactor, F34, F43, f_arr);
    absgq0      = abs(gq_arr(1)); 
    idx_BW      = find(abs(gq_arr) <= (1/sqrt(2))*absgq0, 1, 'first'); 
    f_3dB       = f_arr(idx_BW); 


    % Refine the 3dB estimate with Newtonian iterations 
    for idx = 1:5
        % fprintf('f_3dB = %.2f kHz\n', f_3dB/1e3); 
        [gq, dgq_df]    = get_gq(mu_MW, hbar, Q, C0, z0bar, gq_PreFactor, F34, F43, f_3dB); 
        diff_gq         = abs(gq) - (1/sqrt(2))*absgq0; 
        dabsgq_df       = (1/2/abs(gq))*2*real(dgq_df*conj(gq)); 
        diff_f          = diff_gq / dabsgq_df; 
        f_3dB           = f_3dB - diff_f;   
    end
    
    % Target function 
    % R = (f_3dB)*log2(1 + |gq0|^2/|gref|^2)
    gref    = 1.2e-3;
    
    
    % using gradient ascent 
    [gq0, dgq0_df]      = get_gq(mu_MW, hbar, Q, C0, z0bar, gq_PreFactor, F34, F43, 1e3);
    [gq3dB, dgq3dB_df]  = get_gq(mu_MW, hbar, Q, C0, z0bar, gq_PreFactor, F34, F43, f_3dB); 
    
    R               = f_3dB*log2(1+abs(gq0)^2/gref^2); 
    
    dR_df3db        = log2(1+abs(gq0)^2/gref^2); 
    
    factor2         = getTransmittedProbePower0(RAQR_config, "factor2"); 

    % Deal with derivatives associated with A_LO 
    dgq3dB_dALO     = get_dgq_dALO(mu_MW, hbar, Q , C0, w0, z0bar, gq_PreFactor, factor2, RAQR_config, E34, E43, F34, F43, f_3dB); 
    dgq0_dALO       = get_dgq_dALO(mu_MW, hbar, Q , C0, w0, z0bar, gq_PreFactor, factor2, RAQR_config, E34, E43, F34, F43, 1e3); 
    dF_dALO         = (1/2)/abs(gq3dB)*2*real(dgq3dB_dALO*conj(gq3dB)) - (1/sqrt(2))*(1/2)/abs(gq0)*2*real(dgq0_dALO*conj(gq0)); 
    dF_df           = (1/2)/abs(gq3dB)*2*real(dgq3dB_df*conj(gq3dB)) - (1/sqrt(2))*(1/2)/abs(gq0)*2*real(dgq0_df*conj(gq0)); 
    df3dB_dALO      = -dF_dALO/dF_df; 
    
    dR_dALO         = dR_df3db*df3dB_dALO + f_3dB * (1/log(2))*(1/(1+abs(gq0)^2/gref^2))*(2*abs(gq0)/gref^2) * (1/2)/abs(gq0)*2*real(dgq0_dALO*conj(gq0)); 

    % Deal with derivatives associated with Omega_c. 
    dgq3dB_dOmega_c = get_dgq_dOmega_c(mu_MW, hbar, Q, C0, w0, z0bar, gq_PreFactor, factor2, RAQR_config, F34, F43, f_3dB); 
    dgq0_dOmega_c   = get_dgq_dOmega_c(mu_MW, hbar, Q, C0, w0, z0bar, gq_PreFactor, factor2, RAQR_config, F34, F43, 1e3); 
    dF_dOmega_c     = (1/2)/abs(gq3dB)*2*real(dgq3dB_dOmega_c*conj(gq3dB)) - (1/sqrt(2))*(1/2)/abs(gq0)*2*real(dgq0_dOmega_c*conj(gq0)); 
    df3dB_dOmega_c  = -dF_dOmega_c/dF_df; 
    dR_dOmega_c     = dR_df3db*df3dB_dOmega_c + f_3dB * (1/log(2))*(1/(1+abs(gq0)^2/gref^2))*(2*abs(gq0)/gref^2) * (1/2)/abs(gq0)*2*real(dgq0_dOmega_c*conj(gq0)); 

    % Update RAQR parameters 
    lr_ALO          = 1e-10;
    lr_Omega_c      = 4e5; 

    RAQR_config.A_LO = RAQR_config.A_LO + lr_ALO*dR_dALO; 
    % RAQR_config.Omega_c = RAQR_config.Omega_c + lr_Omega_c*dR_dOmega_c; 

    fprintf('Iter = %d, Analysis: R = %.2f kbps, BW = %.2f kHz, gq0 = %.5f mS\n', iter, R/1e3, f_3dB/1e3, abs(gq0)*1e3); 

    if first
        % do nothing 
        
    else
        est_dgq0_dALO =  (gq0-last_gq0)/lastALO_Adjustment; 
        % est_dgq0_dOmega_c = (gq0 - last_gq0)/lastOmega_c_Adjustment; 
        fprintf('gq Derivative comparison: Est %e v.s. Theroetical %e\n', real(est_dgq0_dALO), real(dgq0_dALO));
    end
    

    fprintf('Adjustment: A_LO %.2f mV/m \n\n', lr_ALO*dR_dALO*1e3); 
    % fprintf('Adjustment: Omega_c %.3f kHz \n\n', lr_Omega_c*dR_dOmega_c/1e3/(2*pi)); 
    % The derivative dgq0_dALO is not correct. Please check this. 
    % ZJA comment: We may consider the change of gq_PreFactor as a function of A_LO.
    % Please take this into consideration. 

    
    iter = iter + 1; 
    R_seq(iter)         = R; 
    BW_seq(iter)        = f_3dB; 
    Omega_c_seq(iter)   = RAQR_config.Omega_c; 
    gq_seq(iter)        = gq0;

    if iter >= Nimax || (~first && abs(R-last_R) <= 1)
        break; 
    end

    % lastOmega_c_Adjustment  = lr_Omega_c*dR_dOmega_c; 
    lastALO_Adjustment  = lr_ALO*dR_dALO; 
    last_gq0            = gq0; 
    last_dgq0_dALO      = dgq0_dALO; 
    
    last_R              = R; 
    
    first = false; 

end

%% Visualization 
close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');

fprintf('Optimized A_LO = %.2f mV/m\n', 1e3*RAQR_config.A_LO); 
fprintf('Optimized Omega_c = %.2f *(2pi) MHz\n', RAQR_config.Omega_c/(1e6*2*pi)); 

figure(1); 
yyaxis left; 
plot((1:iter).', R_seq(1:iter)/(1e6)); 
xlabel('Iterations'); 
ylabel('Rate (Mbps)');
yyaxis right; 
plot((1:iter).', BW_seq(1:iter)/(1e3)); 
ylabel('Bandwidth (kHz)'); 
title('Gradient Ascent Optimizer'); 

figure(2); 
yyaxis left; 
plot((1:iter).', mag2db(abs(gq_seq(1:iter))/gref)); 
xlabel('Iterations'); 
ylabel('SNR (dB)');
yyaxis right; 
plot((1:iter).', BW_seq(1:iter)/(1e3)); 
ylabel('Bandwidth (kHz)'); 


figure(3); 
plot(f_arr, mag2db(abs(gq_arr)*1e3)); 
set(gca, 'xscale', 'log'); grid on; 


%% Utils 

function [gq_arr, dgq_df_arr] = get_gq(mu_MW, hbar, Q , C0, z0bar, gq_PreFactor, F34, F43, f_arr)
    
    Nf          = length(f_arr); 
    gq_arr      = zeros(size(f_arr)); 
    dgq_df_arr  = zeros(size(f_arr)); 

    for idx = 1:Nf
        svar        = 1i*2*pi*f_arr(idx); 

        T34         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F34(2:16,2:16) * z0bar; 
        T34         = T34(2);
        T43         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F43(2:16,2:16) * z0bar; 
        T43         = T43(2);
        
        T34_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F34(2:16,2:16) * z0bar; 
        T34_nf      = T34_nf(2);
        T43_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F43(2:16,2:16) * z0bar; 
        T43_nf      = T43_nf(2); 

        Der_T34_f   = (1i*2*pi)*Q*[zeros(1, 15); eye(15)]*(-1)*(inv(svar*eye(15)-C0))^2 * F34(2:16,2:16) * z0bar; 
        Der_T34_f   = Der_T34_f(2); 
        
        Der_T34_nf  = (-1i*2*pi)*Q*[zeros(1, 15); eye(15)]*(-1)*(inv(conj(svar)*eye(15)-C0))^2 * F34(2:16,2:16) * z0bar;
        Der_T34_nf  = Der_T34_nf(2); 
        
        Der_T43_f   = (1i*2*pi)*Q*[zeros(1, 15); eye(15)]*(-1)*(inv(svar*eye(15)-C0))^2 * F43(2:16,2:16) * z0bar; 
        Der_T43_f   = Der_T43_f(2); 
        
        Der_T43_nf  = (-1i*2*pi)*Q*[zeros(1, 15); eye(15)]*(-1)*(inv(conj(svar)*eye(15)-C0))^2 * F43(2:16,2:16) * z0bar; 
        Der_T43_nf  = Der_T43_nf(2); 

        
        gq_arr(idx)     = gq_PreFactor * (mu_MW/(2*hbar))*((T34-conj(T34_nf))/(2i)+(T43-conj(T43_nf))/(2i)); 
        dgq_df_arr(idx) = gq_PreFactor * (mu_MW/(2*hbar))*((Der_T34_f - conj(Der_T34_nf))/(2i) + (Der_T43_f - conj(Der_T43_nf))/(2i));

    end

end


function [dgq_dALO_arr] = get_dgq_dALO(mu_MW, hbar, Q , C0, w0, z0bar, gq_PreFactor, factor2, RAQR_config, E34, E43, F34, F43, f_arr)

    Nf              = length(f_arr); 
    dgq_dALO_arr    = zeros(size(f_arr)); 

    for idx = 1:Nf
        svar        = 1i*2*pi*f_arr(idx); 

        % Evaluate derivative of (gq) w.r.t. RF LO amplitude A_LO.  
        Der_A0_ALO  = (mu_MW/(2*hbar))*(E34+E43); 
        Der_B0_ALO  = Q'*Der_A0_ALO*Q; 
        Der_w0_ALO  = Der_B0_ALO(2:16, 1); 
        Der_C0_ALO  = Der_B0_ALO(2:16, 2:16); 
        
        Der_z0bar_ALO   = (-1/2)*( (-inv(C0)*Der_C0_ALO*inv(C0)) * w0 + inv(C0)*Der_w0_ALO ); 
        
        Der_T43_ALO     = Q*[zeros(1, 15); eye(15)] * (inv(svar*eye(15)-C0) *(Der_C0_ALO)* inv(svar*eye(15)-C0) * F43(2:16,2:16)*z0bar + inv(svar*eye(15)-C0)*F43(2:16,2:16)*Der_z0bar_ALO); 
        Der_T43_ALO     = Der_T43_ALO(2); 
        
        Der_T34_ALO     = Q*[zeros(1, 15); eye(15)] * (inv(svar*eye(15)-C0) *(Der_C0_ALO)* inv(svar*eye(15)-C0) * F34(2:16,2:16)*z0bar + inv(svar*eye(15)-C0)*F34(2:16,2:16)*Der_z0bar_ALO); 
        Der_T34_ALO     = Der_T34_ALO(2);
        
        % Negate the frequency 
        Der_T43_ALO_nf  = Q*[zeros(1, 15); eye(15)] * (inv(-svar*eye(15)-C0) *(Der_C0_ALO)* inv(-svar*eye(15)-C0) * F43(2:16,2:16)*z0bar + inv(-svar*eye(15)-C0)*F43(2:16,2:16)*Der_z0bar_ALO); 
        Der_T43_ALO_nf  = Der_T43_ALO_nf(2); 
        
        Der_T34_ALO_nf  = Q*[zeros(1, 15); eye(15)] * (inv(-svar*eye(15)-C0) *(Der_C0_ALO)* inv(-svar*eye(15)-C0) * F34(2:16,2:16)*z0bar + inv(-svar*eye(15)-C0)*F34(2:16,2:16)*Der_z0bar_ALO); 
        Der_T34_ALO_nf  = Der_T34_ALO_nf(2);
        
        % use the bootstrap technique to evaluate d(preFactor)/d(A_LO). 
        svar        = 1i*2*pi*1e3; % set to some low-freq. 
        T34         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F34(2:16,2:16) * z0bar; 
        T34         = T34(2);
        T43         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F43(2:16,2:16) * z0bar; 
        T43         = T43(2);
        
        T34_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F34(2:16,2:16) * z0bar; 
        T34_nf      = T34_nf(2);
        T43_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F43(2:16,2:16) * z0bar; 
        T43_nf      = T43_nf(2); 

        gq          = gq_PreFactor * (mu_MW/(2*hbar))*((T34-conj(T34_nf))/(2i)+(T43-conj(T43_nf))/(2i)); 

        dgq_dALO_arr(idx)   = gq_PreFactor* (mu_MW/(2*hbar)) *( (Der_T34_ALO-conj(Der_T34_ALO_nf))/(2i) + (Der_T43_ALO-conj(Der_T43_ALO_nf))/(2i) ); 

        dgq_dALO_arr(idx) = dgq_dALO_arr(idx) + factor2* (gq*RAQR_config.d) * (gq/gq_PreFactor) ; 
        
    end

end

% Get the derivative of gq w.r.t. control light Rabi frequency 
function [dgq_dOmega_c_arr] = get_dgq_dOmega_c(mu_MW, hbar, Q , C0, w0, z0bar, gq_PreFactor, factor2, RAQR_config, F34, F43, f_arr)

    Nf                  = length(f_arr); 
    dgq_dOmega_c_arr    = zeros(size(f_arr)); 

    for idx = 1:Nf
        svar        = 1i*2*pi*f_arr(idx); 

        DM         = zeros(4); 
        DM(2,3)    = 1/2; % half-unit change in Omega_c. 
        
        E23_c      = -1i*(kron(eye(4),DM) - kron(DM.',eye(4))); 
        E32_c      = -1i*(kron(eye(4),DM.') - kron(DM,eye(4))); 
        F23_c      = Q'*E23_c*Q; 
        F32_c      = Q'*E32_c*Q; 

        
        Der_A0_Omega_c      = (E23_c+E32_c); 
        Der_B0_Omega_c      = Q'*Der_A0_Omega_c*Q; 
        Der_w0_Omega_c      = Der_B0_Omega_c(2:16, 1); 
        Der_C0_Omega_c      = Der_B0_Omega_c(2:16, 2:16); 
        Der_z0bar_Omega_c   = (-1/2)*( (-inv(C0)*Der_C0_Omega_c*inv(C0)) * w0 + C0\Der_w0_Omega_c ); 
        
        Der_T43_Omega_c     = Q*[zeros(1, 15); eye(15)] * (inv(svar*eye(15)-C0) *(Der_C0_Omega_c)* inv(svar*eye(15)-C0) * F43(2:16,2:16)*z0bar + inv(svar*eye(15)-C0)*F43(2:16,2:16)*Der_z0bar_Omega_c); 
        Der_T43_Omega_c     = Der_T43_Omega_c(2); 
        
        Der_T34_Omega_c     = Q*[zeros(1, 15); eye(15)] * (inv(svar*eye(15)-C0) *(Der_C0_Omega_c)* inv(svar*eye(15)-C0) * F34(2:16,2:16)*z0bar + inv(svar*eye(15)-C0)*F34(2:16,2:16)*Der_z0bar_Omega_c); 
        Der_T34_Omega_c     = Der_T34_Omega_c(2);
        
        Der_T43_Omega_c_nf  = Q*[zeros(1, 15); eye(15)] * (inv(-svar*eye(15)-C0) *(Der_C0_Omega_c)* inv(-svar*eye(15)-C0) * F43(2:16,2:16)*z0bar + inv(-svar*eye(15)-C0)*F43(2:16,2:16)*Der_z0bar_Omega_c); 
        Der_T43_Omega_c_nf  = Der_T43_Omega_c_nf(2); 
        
        Der_T34_Omega_c_nf  = Q*[zeros(1, 15); eye(15)] * (inv(-svar*eye(15)-C0) *(Der_C0_Omega_c)* inv(-svar*eye(15)-C0) * F34(2:16,2:16)*z0bar + inv(-svar*eye(15)-C0)*F34(2:16,2:16)*Der_z0bar_Omega_c); 
        Der_T34_Omega_c_nf  = Der_T34_Omega_c_nf(2);


        T34         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F34(2:16,2:16) * z0bar; 
        T34         = T34(2);
        T43         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F43(2:16,2:16) * z0bar; 
        T43         = T43(2);
        
        T34_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F34(2:16,2:16) * z0bar; 
        T34_nf      = T34_nf(2);
        T43_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F43(2:16,2:16) * z0bar; 
        T43_nf      = T43_nf(2); 
        

        % use the bootstrap technique to evaluate d(preFactor)/d(Omega_c). 
        svar        = 1i*2*pi*1e3; % set to some low-freq. 
        T23         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F23_c(2:16,2:16) * z0bar; 
        T23         = T23(2);
        T32         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F32_c(2:16,2:16) * z0bar; 
        T32         = T32(2);
        
        T23_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F23_c(2:16,2:16) * z0bar; 
        T23_nf      = T23_nf(2);
        T32_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F32_c(2:16,2:16) * z0bar; 
        T32_nf      = T32_nf(2); 

        drho_dOmega_c       = ((T23-conj(T23_nf))/(2i)+(T32-conj(T32_nf))/(2i)); 
        % change in Im{rho_{21}} w.r.t. change in \Omega_c, evaluated at freq 1kHz@control. This is approximately a real value. 
        
        dPreFactor_dOmegac       = gq_PreFactor*(RAQR_config.d*factor2)*drho_dOmega_c; 
        % ZJA: Each time we take the derivative of something that contains
        % Pbar, we just multiply it with (RAQR_config.d*factor2), and then
        % multiply the derivative of Im{rho_{21}} w.r.t. anything that we
        % are interested in. 

        dgq_dOmega_c_arr(idx)    = gq_PreFactor * (mu_MW/(2*hbar))*( (Der_T34_Omega_c-conj(Der_T34_Omega_c_nf))/(2i) + (Der_T43_Omega_c-conj(Der_T43_Omega_c_nf))/(2i) ); 

        dgq_dOmega_c_arr(idx)    = dgq_dOmega_c_arr(idx) + dPreFactor_dOmegac * (mu_MW/(2*hbar)) * ((T34-conj(T34_nf))/(2i)+(T43-conj(T43_nf))/(2i)); 
        % Now only god knows whether it is correct. 

    end


end



