function atomic_response = transientQuantumSimulation(RAQR_config, sim_config, inputEfield, init_method)
    % Physical constants
    epsilon_0   = 8.85e-12; 
    hbar        = 6.626e-34 / (2*pi); 

    a0      = 5.2918e-11; 
    e       = 1.6e-19; 
    mu_MW   = 1443.45*e*a0; 
    mu_12   = 4.5022*e*a0; 

    d = RAQR_config.d; 
    N0 = RAQR_config.N0; 

    if ~exist('init_method', 'var') || isempty(init_method)
        init_method = "SteadyState"; 
    end

    
%% Extract RAQR configurations
    Delta_p = RAQR_config.Delta_p; 
    Delta_c = RAQR_config.Delta_c; 
    Delta_l = RAQR_config.Delta_l; 
    
    gamma   = RAQR_config.gamma; 
    gamma_2 = RAQR_config.gamma_2; 
    gamma_3 = RAQR_config.gamma_3; 
    gamma_4 = RAQR_config.gamma_4; 
    gamma_c = RAQR_config.gamma_c; 
    
    
    Omega_c = RAQR_config.Omega_c; 
    Omega_p = RAQR_config.Omega_p; 
    lambda_c = RAQR_config.lambda_c; 
    lambda_p = RAQR_config.lambda_p; 
    
    % Compute Rabi frequencies for the RF input sequence 
    Omega_RFseq    = mu_MW*(inputEfield)/hbar;      

    % Set-up transient simulation parameters 

    Temperature = sim_config.Temperature; 

    NA      = 6.02e23; 
    M_Cs    = 132.9; 
    m_Cs    = (M_Cs/1e3)/NA; 
    kB      = getPhysicalConstant('Boltzmann'); 

    sigma_vx    = sqrt(kB*Temperature/m_Cs);  
    Nd          = 1501; 
    Nd_mid      = (Nd+1)/2; 
    
   
    [x_knots, vx_probs] = getNormalQuadrature(Nd); 
    vx_knots = sigma_vx * x_knots; 
    
    
    
%% Solve the master equation with Runge-Kutta(4)-method 
    
    % Step 1: Initiate density matrices to their steady-state 
    rhovec_init = zeros(16, Nd); 
    
    if init_method == "SteadyState" 
        for idx_v = 1:Nd
            vx      = vx_knots(idx_v); 
            delta_p = Delta_p + 2*pi*vx/lambda_p; 
            delta_c = Delta_c - 2*pi*vx/lambda_c; 
            delta_l = Delta_l; 
            omega_p = Omega_p; omega_c = Omega_c; 
            Omega_RF = Omega_RFseq(1); 
        
            H0= [0,         conj(omega_p)/2,    0,                      0;                  ...
                omega_p/2,  -delta_p,           conj(omega_c)/2,        0;                  ...
                0,          omega_c/2,          -delta_p-delta_c,       conj(Omega_RF)/2;   ...
                0,          0,                  (Omega_RF)/2,           -delta_p-delta_c-delta_l]; 
            
            Gamma = diag([gamma, gamma+gamma_2, gamma+gamma_3+gamma_c, gamma+gamma_4]); 
            
            A0 = -1i*(kron(eye(4), H0) - kron(H0.', eye(4))) - (1/2)*(kron(Gamma, eye(4)) + kron(eye(4), Gamma)); 
            A0(1, 6)    = A0(1, 6) + gamma_2; 
            A0(1, 16)   = A0(1, 16) + gamma_4; 
            A0(6, 11)   = A0(6, 11) + gamma_3; 
        
            % Re-polulation 
            A0(1,1) = A0(1,1) + gamma; 
            A0(1,6) = A0(1,6) + gamma; 
            A0(1,11) = A0(1,11) + gamma; 
            A0(1,16) = A0(1,16) + gamma; 
            
            % Sanity check 
            % Find the steady state solution x_ss that corresponds to the zero-eigenvector of A0, and ensure that <u4, x_ss> ==1
            % 
            N           = norm(A0, 'fro'); 
            u4          = N*eye(4); u4 = u4(:); 
            rhovec      = (A0'*A0 + u4*u4')\(N*u4);   % [Reference] 
            rhovec_init(:, idx_v) = rhovec; 
            
            if ~ishermitian(reshape(rhovec, [4,4])) 
                rho = reshape(rhovec, [4,4]); 
                rho = (rho + rho')/2; 
                rhovec_init(:, idx_v) = rho(:); 
            end
            
        end

    elseif init_method == "GroundState" 
        rhovec_init(1, :) = ones(1, Nd); 
    
    else 
        error('density matrix initialization method %s not implemented.\n', init_method); 

    end
    
    
    % Step 2: Perform Runge-Kutta for each initial value and each vx 
    Nt          = sim_config.Nt;
    rho21_mat   = zeros(Nd, Nt); 
    rhovec_final = zeros(16, Nd); 


    dataQueue = parallel.pool.DataQueue; 
    progressDisp(Nd);
    afterEach(dataQueue, @progressDisp); 
    
    parfor idx_v = 1:Nd
        vx      = vx_knots(idx_v); 
        delta_p = Delta_p + 2*pi*vx/lambda_p; 
        delta_c = Delta_c - 2*pi*vx/lambda_c; 
        delta_l = Delta_l; 
        omega_p = Omega_p; omega_c = Omega_c; 
        
        % Re-determine the Runge-Kutta step size 
        omega_max = max(abs([delta_p, delta_c, delta_l, omega_p, omega_c])); 
    
        if 2*pi/(2.5*sim_config.dt) < omega_max
            finer       = true; 
            dt          = 2*pi/omega_max / 2.5; 
            t_arr       = 0.0 : dt : sim_config.T; 
            Nrk         = length(t_arr); 
            
            inner_rho_arr       = zeros(16, Nrk); 
            inner_rho_arr(:, 1) = rhovec_init(:, idx_v); 
            inner_Omega_RFseq   = interp1(sim_config.t_arr, Omega_RFseq, t_arr, "linear", "extrap"); 
    
        else
            finer   = false;
            Nrk     = Nt;
            t_arr   = sim_config.t_arr; 
            dt      = sim_config.dt;
    
            inner_rho_arr         = zeros(16, Nt); 
            inner_rho_arr(:, 1)   = rhovec_init(:, idx_v); 
            inner_Omega_RFseq     = Omega_RFseq; 
        end
        
        for idx_t = 1:(Nrk-1)
            rho_i       = reshape(inner_rho_arr(:, idx_t), [4, 4]); 
            Omega_RF    = inner_Omega_RFseq(idx_t); 
    
            % Hamiltonian, relaxation an decay matrices 
            Lambda  = diag([gamma+gamma_2*rho_i(2,2)+gamma_4*rho_i(4,4), gamma_3*rho_i(3,3), 0, 0]); 
    
            
            %  |1> 6S1/2    |2> 6P3/2   |3> 47D5/2          |4> 48P3/2 
            H = [0,         omega_p/2,  0,                      0;                  ...
                omega_p/2,  -delta_p,    omega_c/2,             0;                  ...
                0,          omega_c/2,  -delta_p-delta_c,       conj(Omega_RF)/2;   ...
                0,          0,          Omega_RF/2,            -delta_p-delta_c-delta_l]; 
            
            Gamma = diag([gamma, gamma+gamma_2, gamma+gamma_3+gamma_c, gamma+gamma_4]); 
    
        
            k1      = dt*((-1i)*(H*rho_i-rho_i*H) - (1/2)*(Gamma*rho_i+rho_i*Gamma) + Lambda); 
            rho_ik1 = rho_i + (1/2)*k1; 
        
            k2      = dt*((-1i)*(H*rho_ik1-rho_ik1*H) - (1/2)*(Gamma*rho_ik1+rho_ik1*Gamma) + Lambda); 
            rho_ik2 = rho_i + (1/2)*k2; 
            
            k3      = dt*((-1i)*(H*rho_ik2-rho_ik2*H) - (1/2)*(Gamma*rho_ik2+rho_ik2*Gamma) + Lambda); 
            rho_ik3 = rho_i + k3; 
            
            k4      = dt*((-1i)*(H*rho_ik3-rho_ik3*H) - (1/2)*(Gamma*rho_ik3+rho_ik3*Gamma) + Lambda); 
        
            inner_rho_arr(:, idx_t+1) = inner_rho_arr(:, idx_t) + (k1(:)+2*k2(:)+2*k3(:)+k4(:))/6; 
            
            % trace(reshape(inner_rho_arr(:, idx_t+1), [4,4])) - 1 
            
        end
        
        if ~finer
            rho21_mat(idx_v, :) = inner_rho_arr(2, :); 
        else
            rho21_mat(idx_v, :) = interp1(t_arr, inner_rho_arr(2,:), sim_config.t_arr, "linear", "extrap"); 
        end
        
        rhovec_final(:, idx_v) = inner_rho_arr(:, end);  
        % fprintf('Doppler class %d/%d complete.\n', idx_v, Nd);  
        send(dataQueue, struct('idx', idx_v, 'val', 1, 'trace', real(trace(rho_i))));  
    end
    
    rho21_seq = vx_probs * rho21_mat; 
    
    atomic_response.rho21_seq       = rho21_seq; 
    atomic_response.chi_e           = -2*(N0*mu_12^2)/(epsilon_0*hbar*Omega_p)*rho21_seq; 
    atomic_response.probeResponse   = (20/log(10))*(-pi*d/lambda_p)*imag(-(2*N0*mu_12^2)/(epsilon_0*hbar*Omega_p)*rho21_seq); 
    atomic_response.rho21_mid       = rho21_mat(Nd_mid, :); 
    atomic_response.rhovec_final    = rhovec_final; 

end 


function progressDisp(arg)
    persistent progArray;
    persistent N; 
    persistent prevDisp; 
    eps         = 1e-6; 

    if isstruct(arg)
        progArray(arg.idx) = arg.val;
        assert(abs(arg.trace-1) <= eps); 

        if prevDisp > 0
            if isunix()
                clc;
            else
                fprintf(repmat('\b', [1, prevDisp]));     % The '\b' backspace works on Windows systems, but not on Linux. 
            end
        end
       
        prevDisp = fprintf('Total progress: %.2f %%\n', sum(progArray)/N*100); 

    else
        progArray = zeros(arg, 1);
        N = arg;
        prevDisp = 0;
    end

end
