function gq_struct = getAnalyticalFreqResponse(RAQR_config, Temperature, freq_arr)

    epsilon_0   = 8.85e-12; 
    mu_0        = 4*pi*1e-7; 
    hbar        = 6.626e-34 / (2*pi); 
    c0          = physconst('LightSpeed'); 
    
    kB      = getPhysicalConstant("Boltzmann"); 
    a0      = 5.2918e-11; 
    e       = 1.6e-19; 
    mu_MW   = 1443.45*e*a0; 
    mu_12   = 4.5022*e*a0;  
    eps     = 1e-6; 
        
    
    kp      = 2*pi/(RAQR_config.lambda_p); 
    kc      = 2*pi/(RAQR_config.lambda_c);  
    
    
    delta_p = RAQR_config.Delta_p;
    delta_c = RAQR_config.Delta_c;
    delta_l = RAQR_config.Delta_l; 
    
    gamma_2 = RAQR_config.gamma_2; 
    gamma_3 = RAQR_config.gamma_3; 
    gamma_4 = RAQR_config.gamma_4; 
    gamma   = RAQR_config.gamma; 

    omega_p = RAQR_config.Omega_p; 
    omega_c = RAQR_config.Omega_c; 
    Omega_RF = mu_MW*RAQR_config.A_LO/hbar; 
    
    
    H0= [0,         conj(omega_p)/2,    0,                      0;                  ...
        omega_p/2,  -delta_p,           conj(omega_c)/2,        0;                  ...
        0,          omega_c/2,          -delta_p-delta_c,       conj(Omega_RF)/2;   ...
        0,          0,                  (Omega_RF)/2,           -delta_p-delta_c-delta_l]; 
    
    % Dephasing matrix Gamma 
    Gamma = diag([0, gamma_2, gamma_3, gamma_4]+gamma); 
    
    % Evolution matrix A0 
    A0 = -1i*(kron(eye(4), H0) - kron(H0.', eye(4))) - (1/2)*(kron(Gamma, eye(4)) + kron(eye(4), Gamma)); 
    A0(1, 6)    = A0(1, 6) + gamma_2; 
    A0(1, 16)   = A0(1, 16) + gamma_4; 
    A0(6, 11)   = A0(6, 11) + gamma_3; 

    % Re-population 
    A0(1,1) = A0(1,1) + gamma; 
    A0(1,6) = A0(1,6) + gamma; 
    A0(1,11) = A0(1,11) + gamma; 
    A0(1,16) = A0(1,16) + gamma; 
    
    %% Find the zero-Doppler steady state solution x_ss
    % It corresponds to the zero-eigenvector of A0, and ensure that <u4, x_ss> ==1
    N       = norm(A0, 'fro'); 
    u4      = N*eye(4); u4 = u4(:); 
    x_ss    = (A0'*A0 + u4*u4')\(N*u4);   % [Reference] 
    rho_ss  = reshape(x_ss, [4, 4]); 
    assert(abs(trace(rho_ss)-1) <= eps && norm(rho_ss-rho_ss', 'fro') <= eps); 
    
    % Consider Doppler effects 
    NA          = 6.02e23; 
    M_Cs        = 132.9; 
    m_Cs        = (M_Cs/1e3)/NA; 
    sigma_vx    = sqrt(kB*Temperature/m_Cs);  
    
    % Av has dimension [Hz]. 
    Av          = (2*pi*sigma_vx/RAQR_config.lambda_p)*(-1i)*(kron(eye(4), diag([0, -1, -1, -1])) - kron(diag([0, -1, -1, -1]).', eye(4))); 
    Av          = Av + (-2*pi*sigma_vx/RAQR_config.lambda_c)*(-1i)*(kron(eye(4), diag([0, 0, -1, -1])) - kron(diag([0, 0, -1, -1]).', eye(4))); 
    

    %% Compute the Doppler-averaged steady-state density matrix rho_Dss.  
    [V, D] = eig(A0); 
    [~, order] = sort(abs(diag(D)), 'ascend'); 
    V = V(:, order);  % Note that the eigenvectors of A0 are not necessarily orthogonal. 
    d = diag(D); 
    d = d(order); 

    % Compute the Moore-Penrose pseudo-inverse with the eigen-decomposition method 
    A0_pinv = V*diag([0; 1./d(2:end)]) / V;  
    [lambdas_G, R, L] = getEigenDecomposition((A0_pinv)*Av); 
    
    vec_rho_d = zeros(16, 1); 
    for idx = 1:16 
        lam = lambdas_G(idx);
        
        if abs(lam) < eps
            coef = 1;
        else
            coef = (-1/lam)*getJ(-1/lam); % The function J(z) is defined in my paper.  
        end
    
        vec_rho_d = vec_rho_d + coef * R(:,idx) * (L(idx, :)*rho_ss(:)); 
    end
    rho_Dss = reshape(vec_rho_d, [4,4]); 
    assert(abs(trace(rho_Dss)-1) <= eps); 


    %% Construct the orthogonal transform matrices to make C0 non-singular 
    Qtmp        = eye(4^2); 
    Qtmp(:,1)   = (u4.')/norm(u4); 
    [Q, ~]      = qr(Qtmp); 
    Q(:,1)      = -Q(:, 1); 
    
    B0      = Q'*A0*Q; 
    C0      = B0(2:end, 2:end); 
    w0      = B0(2:end, 1); 
    assert(norm(B0(1,:)) <= eps); 
    
    Bv      = Q'*Av*Q; 
    Cv      = Bv(2:end, 2:end); 
    wv      = Bv(2:end, 1); 
    assert(norm(Bv(1,:))<=eps && norm(wv)<=eps); % We can prove that Av(1,1) = 0. 
    
    
    y_1bar  = 1/sqrt(4);  
    z0bar   = -C0\(y_1bar*w0); 
    % x0bar   = Q*[y_1bar; z0bar];            % Steady-state x solution 
    % rho_bar = reshape(x0bar, [4,4]);        % Steady-state density matrix solution 
    
    
    %% Compute the analytical freq. response for all the frequencies. 
    assert(isrow(freq_arr)); 
    Nf          = length(freq_arr); 
    dImrho21_dE = zeros(1, Nf); 
    
    parfor idx_f = 1:Nf

        svar = 1i*2*pi*freq_arr(idx_f);  
        
        [Lambda_s, R_s, L_s]    = getEigenDecomposition((svar*eye(15)-C0)\Cv);
        [Lambda_ns, R_ns, L_ns] = getEigenDecomposition((-svar*eye(15)-C0)\Cv);
        [Lambda_0, R_0, L_0]    = getEigenDecomposition(-C0\Cv); 
        
        I2mat   = zeros(15); 
        I2mat_n = zeros(15); 
        
        for mm = 1:15
            for nn = 1:15
                if abs(Lambda_s(mm)) <= eps
                    if abs(Lambda_0(nn)) <= eps
                        I2mat(mm, nn) = 1; 
                    else
                        I2mat(mm, nn) = (1/Lambda_0(nn)) * getJ(1/Lambda_0(nn)); 
                    end
                else
                    if abs(Lambda_0(nn)) <= eps
                        I2mat(mm, nn) = (1/Lambda_s(mm)) * getJ(1/Lambda_s(mm)); 
                    else
                        % E[1/(1-zX)] => (z^{-1}J(z^{-1}) 
                        if abs(Lambda_s(mm) - Lambda_0(nn)) <= eps
                            warning('Numerically unstable.\n'); 
                        end
                        I2mat(mm, nn) = (getJ(1/Lambda_0(nn)) - getJ(1/Lambda_s(mm)))/(Lambda_0(nn) - Lambda_s(mm)); 
                        
                    end
                end
            end
        end
        
        for mm = 1:15
            for nn = 1:15
                if abs(Lambda_ns(mm)) <= eps
                    if abs(Lambda_0(nn)) <= eps
                        I2mat_n(mm, nn) = 1;
                    else
                        I2mat_n(mm, nn) = (1/Lambda_0(nn)) * getJ(1/Lambda_0(nn)); 
                    end
                else
                    if abs(Lambda_0(nn)) <= eps
                        I2mat_n(mm, nn) = (1/Lambda_ns(mm)) * getJ(1/Lambda_ns(mm)); 
                    else
                        % E[1/(1-zX)] => (z^{-1}J(z^{-1}) 
                        I2mat_n(mm, nn) = (getJ(1/Lambda_0(nn)) - getJ(1/Lambda_ns(mm)))/(Lambda_0(nn) - Lambda_ns(mm)); 
                    end
                end
            end
        end
        
        
        H43     = zeros(4); H43(4,3) = 1; 
        E43     = -1i*(kron(eye(4), H43) - kron(H43.', eye(4)));
        E34     = -1i*(kron(eye(4), H43.') - kron(H43, eye(4)));
        F43     = Q'*E43*Q; 
        F34     = Q'*E34*Q; 
        
        % Evaluate DopplerMean(T43(s)). 
        temp = (svar*eye(15)-C0)\F43(2:end, 2:end); 
        T43 = zeros(16, 1); 
        for mm = 1:15
            for nn = 1:15
                uu      = R_s(:, mm)*L_s(mm, :); 
                vv      = R_0(:, nn)*L_0(nn, :); 
                T43     = T43 + I2mat(mm, nn)*Q*[zeros(1, 15); eye(15)] * uu * temp * vv * z0bar; 
            end
        end
        T43 = T43(2); 
        
        % Evaluate DopplerMean(T34(s)). 
        temp = (svar*eye(15)-C0)\F34(2:end, 2:end); 
        T34 = zeros(16, 1); 
        for mm = 1:15
            for nn = 1:15
                uu = R_s(:, mm)*L_s(mm, :); 
                vv = R_0(:, nn)*L_0(nn, :); 
                T34 = T34 + I2mat(mm, nn)*Q*[zeros(1, 15); eye(15)] * uu * temp * vv * z0bar; 
            end
        end
        T34 = T34(2); 
        
        % Evaluate DopplerMean(T43(s*)) 
        temp    = (-svar*eye(15)-C0)\F43(2:end, 2:end); 
        T43_ns  = zeros(16, 1); 
        for mm = 1:15
            for nn = 1:15
                uu = R_ns(:, mm)*L_ns(mm, :); 
                vv = R_0(:, nn)*L_0(nn, :); 
                T43_ns = T43_ns + I2mat_n(mm, nn)*Q*[zeros(1, 15); eye(15)] * uu * temp * vv * z0bar; 
            end
        end
        T43_ns = T43_ns(2); 
        
        % Evaluate DopplerMean(T34(s*)) 
        temp    = (-svar*eye(15)-C0)\F34(2:end, 2:end); 
        T34_ns  = zeros(16, 1); 
        for mm = 1:15
            for nn = 1:15
                uu      = R_ns(:, mm)*L_ns(mm, :); 
                vv      = R_0(:, nn)*L_0(nn, :); 
                T34_ns  = T34_ns + I2mat_n(mm, nn)*Q*[zeros(1, 15); eye(15)] * uu * temp * vv * z0bar; 
            end
        end
        T34_ns = T34_ns(2); 
        
        dImrho21_dE(idx_f) = (mu_MW/(2*hbar)) * ((T34-conj(T34_ns))/(2i)+(T43-conj(T43_ns))/(2i)); 
    end

    %% Post-processing

    alpha_0             = -(kp*RAQR_config.N0*(mu_12^2))/(epsilon_0*hbar*omega_p)*imag(rho_Dss(2,1)); 
    P_probe             = 29.8e-6; 
    probeTransmission   = exp(-2*alpha_0*RAQR_config.d); 
    Iph0                = (RAQR_config.eta*P_probe)/(hbar*(c0*kp))*e*probeTransmission; 
    gq_PreFactor        = Iph0*(2*kp*RAQR_config.N0*(mu_12^2))/(epsilon_0*hbar*omega_p); 
    gq_arr              = gq_PreFactor * dImrho21_dE; 


    gq_struct.dImrho21_dE = dImrho21_dE; 
    gq_struct.gq          = gq_arr; 
    gq_struct.pt          = probeTransmission; 

end
