function [Tq, dTq, rho_bar] = getDCTransferCoeffs(RAQR_config, M, mu_arr, gamma_arr, delta_arr, A_LO_arr)
% ================================================================================== %
% INPUT PARAMETERS: 
% M:               Number of bands;
% mu_arr(1:M):     (COL VECTOR) Transition dipole of each Rydberg transition that RF-resonates with |3>;  
% gamma_arr(1:M):  (COL VECTOR) Dephasing coefficient of each Rydberg level that RF-resonates with |3>; 
% delta_arr(1:M):  (COL VECTOR) Detuning angular frequencies of each RF-LO; 
% A_LO_arr(1:M):   (COL VECTOR) LO Amplitude; 
% 
% OUTPUT PARAMETERS: 
% Tq:              (COL VECTOR) DC Transfer coefficients from A_LO{1,2,...,M} to rho_{21}
% dTq:             The Jacobi matrix of the above R^M->C^M mapping 
% rho_bar:         Steady-state density matrix 
% ================================================================================== %

assert(length(gamma_arr) == M); 
assert(length(delta_arr) == M); 
assert(length(A_LO_arr) == M); 

hbar            = getPhysicalConstant("PlanckConstant")/(2*pi); 
Omega_LO_arr    = mu_arr.*A_LO_arr/hbar; 
Omega_p         = RAQR_config.Omega_p; 
Omega_c         = RAQR_config.Omega_c; 


tmp             = cumsum([0; -RAQR_config.Delta_p; -RAQR_config.Delta_c; -delta_arr]); 
H               = diag(tmp); 
H(1,2)          = Omega_p/2; 
H(2,1)          = Omega_p/2; 
H(2,3)          = Omega_c/2; 
H(3,2)          = Omega_c/2; 

for m = 1:M
    H(3, m+3) = Omega_LO_arr(m)/2; 
    H(m+3, 3) = Omega_LO_arr(m)/2; 
end
Gamma = diag([0; RAQR_config.gamma_2; RAQR_config.gamma_3; gamma_arr]); 

didx    = zeros(1, 3+M); 
for idx = 1:(3+M)
    didx(idx) = (idx-1)*(3+M)+idx; 
end

A0                      = -1i*(kron(eye(M+3), H) - kron(H.', eye(M+3))) - (1/2)*(kron(eye(M+3), Gamma) + kron(Gamma, eye(M+3))); 
A0(didx(1), didx(2))    = A0(didx(1), didx(2)) + RAQR_config.gamma_2; 
A0(didx(2), didx(3))    = A0(didx(2), didx(3)) + RAQR_config.gamma_3;

for m = 1:M
    A0(didx(1), didx(m+3))    = A0(didx(1), didx(m+3)) + gamma_arr(m); 
end

dAMat = cell(M,1); 
for m = 1:M
    HAct            = zeros(M+3); 
    HAct(3,m+3)     = 1; 
    HAct(m+3,3)     = 1; 
    dAMat{m}        = -1i*(kron(eye(M+3), HAct) - kron(HAct.', eye(M+3))); 
end

eps     = 1e-6; 
u       = zeros(1, (M+3)^2); 
u(didx) = 1; 
assert(norm(u*A0) <= eps, "A0 matrix error: Trace-preserving property not satisfied.\n");

% Solve the steady state of the system 
Qtmp        = eye((M+3)^2); 
Qtmp(:,1)   = (u.')/norm(u); 
[Q, ~]      = qr(Qtmp); 
Q(:,1)      = sign(u*Q(:,1))*Q(:, 1); 

B0      = Q'*A0*Q; 
C0      = B0(2:end, 2:end); 
w0      = B0(2:end, 1); 
assert(norm(B0(1,:)) <= eps); 

y_1bar  = 1/sqrt(M+3);  
z0bar   = -C0\(y_1bar*w0); 
x0bar   = Q*[y_1bar; z0bar];                % Steady-state x solution 
rho_bar = reshape(x0bar, [M+3,M+3]);        % Steady-state density matrix solution 


% Linear equation setup: C_0 z_0 + (y_1bar)*w0 == 0. 
dCmat   = cell(M,1);
dwmat   = cell(M,1); 
for m = 1:M
    dBm         = Q'*dAMat{m}*Q; 
    dCmat{m}    = dBm(2:end, 2:end); 
    dwmat{m}    = dBm(2:end, 1); 

    assert(norm(dBm(1,:)) <= eps); 
    assert(norm(dBm(2:end, 1)) <= eps); 
end

Tq      = zeros(M, 1); 
dzmat   = cell(M,1); 

for m = 1:M
    dz      = -C0\(y_1bar*dwmat{m}+dCmat{m}*z0bar); 
    dzmat{m}= dz; 
    dx      = Q*[0; dz]; 
    Tq(m)   = dx(2); 
end

dTq = zeros(M); 
for ii = 1:M
    for jj = 1:M
        d2z         = C0\ ((dCmat{jj}* (C0\(y_1bar*dwmat{ii}+dCmat{ii}*z0bar)) ) - (dCmat{ii}*dzmat{jj})); 
        d2x         = Q*[0; d2z]; 
        dTq(ii, jj) = d2x(2); 
    end
end

end
