function gq_struct = getNumericalFreqResponse(RAQR_config, Temperature, freq_arr)
    % output gq_arr: row vector 
    
    kB          = getPhysicalConstant("Boltzmann"); 
    e           = 1.6e-19; 
    a0          = 5.2918e-11; 
    hbar        = 6.626e-34 / (2*pi); 
    epsilon_0   = 8.85e-12;
    
    
    mu_RF       = 1443.45*e*a0;  
    mu_12       = 4.5022*e*a0; 
    c0          = physconst('LightSpeed');
    kp          = 2*pi/(RAQR_config.lambda_p); 


%% Evaluate freq response 
    Omega_c     = RAQR_config.Omega_c; 
    Omega_p     = RAQR_config.Omega_p; 
    Omega_RF    = mu_RF*(RAQR_config.A_LO)/hbar;    
    
    gamma       = RAQR_config.gamma;    % Dephasing rate of each state |n> 
    gamma_2     = RAQR_config.gamma_2; 
    gamma_3     = RAQR_config.gamma_3; 
    gamma_4     = RAQR_config.gamma_4; 
    
    lambda_p    = RAQR_config.lambda_p; 
    lambda_c    = RAQR_config.lambda_c; 
    
    
    % Consider Doppler effects 
    NA          = 6.02e23; 
    M_Cs        = 132.9; 
    m_Cs        = (M_Cs/1e3)/NA; 
    sigma_vx    = sqrt(kB*Temperature/m_Cs);  
    Nd          = 1501; 
    Nd_mid      = (Nd+1)/2; 
    [x_knots, vx_probs] = getNormalQuadrature(Nd, "DenseCenter"); 
    vx_knots    = sigma_vx * x_knots; 
    
    
    Nf = length(freq_arr); 
    
    drho21_dE_tmp   = zeros(Nd, Nf); 
    rho21_tmp       = zeros(1, Nd); 

    parfor idx_d = 1:Nd
        Delta_p     = RAQR_config.Delta_p - 2*pi*vx_knots(idx_d)/lambda_p; 
        Delta_c     = RAQR_config.Delta_c + 2*pi*vx_knots(idx_d)/lambda_c;  
        Delta_LO    = RAQR_config.Delta_l; 
        
        
        % Quantum state definition 
        %  |1> 6S1/2    |2> 6P3/2   |3> 47D5/2          |4> 48P3/2                  
        
        H = [0,         Omega_p/2,  0,                  0;                          ...                                                           ...
            Omega_p/2,  -Delta_p,    Omega_c/2,         0;                          ...
            0,          Omega_c/2,  -Delta_p-Delta_c,   Omega_RF/2;                 ...
            0,          0,          Omega_RF/2,         -Delta_p-Delta_c-Delta_LO ]; 
        
        
        Gamma   = diag([0, gamma_2, gamma_3, gamma_4]+gamma); 
        
        didx    = zeros(1, 4); 
        for idx = 1:4
            didx(idx) = (idx-1)*4+idx; 
        end
        
        % A0 and the dephasing terms 
        A0                      = -1i*(kron(eye(4), H) - kron(H.', eye(4))) - (1/2)*(kron(eye(4), Gamma) + kron(Gamma, eye(4))); 
        A0(didx(1), didx(2))    = A0(didx(1), didx(2)) + gamma_2; 
        A0(didx(1), didx(4))    = A0(didx(1), didx(4)) + gamma_4; 
        A0(didx(2), didx(3))    = A0(didx(2), didx(3)) + gamma_3;
        
        % Re-population 
        A0(didx(1), didx(1))    = A0(didx(1), didx(1)) + gamma; 
        A0(didx(1), didx(2))    = A0(didx(1), didx(2)) + gamma; 
        A0(didx(1), didx(3))    = A0(didx(1), didx(3)) + gamma; 
        A0(didx(1), didx(4))    = A0(didx(1), didx(4)) + gamma; 

        
        u4          = zeros(1, 16); 
        u4(didx)    = 1/4;
        eps         = 1e-6;
        
        assert(norm(u4*A0) <= eps, "A0 matrix error: Trace-preserving property not satisfied.\n");
        
        Qtmp        = eye(4^2); 
        Qtmp(:,1)   = (u4.')/norm(u4); 
        [Q, ~]      = qr(Qtmp); 
        Q(:,1)      = -Q(:, 1); 
        
        B0      = Q'*A0*Q; 
        C0      = B0(2:end, 2:end); 
        w0      = B0(2:end, 1); 
        assert(norm(B0(1,:)) <= eps); 
        
        y_1bar  = 1/sqrt(4);  
        z0bar   = -C0\(y_1bar*w0); 
        x0bar   = Q*[y_1bar; z0bar];            % Steady-state x solution 
        rho_bar = reshape(x0bar, [4,4]);        % Steady-state density matrix solution 
        rho21_tmp(idx_d) = rho_bar(2, 1); 

        H43     = zeros(4); H43(4,3) = 1; 
        E43     = -1i*(kron(eye(4), H43) - kron(H43.', eye(4)));
        E34     = -1i*(kron(eye(4), H43.') - kron(H43, eye(4)));
        F43     = Q'*E43*Q; 
        F34     = Q'*E34*Q; 
        
        for idx_f = 1:Nf
            f_IF        = freq_arr(idx_f); 
            svar        = 1i*2*pi*f_IF; 
            
            T34         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F34(2:end,2:end) * z0bar; 
            T34         = T34(2);
            T43         = Q*[zeros(1, 15); eye(15)]* inv(svar*eye(15)-C0) * F43(2:end,2:end) * z0bar; 
            T43         = T43(2);
            
            T34_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F34(2:end,2:end) * z0bar; 
            T34_nf      = T34_nf(2);
            T43_nf      = Q*[zeros(1, 15); eye(15)]* inv(-svar*eye(15)-C0) * F43(2:end,2:end) * z0bar; 
            T43_nf      = T43_nf(2); 
            
            
            drho21_dE_tmp(idx_d, idx_f)    = (mu_RF/(2*hbar)) * ((T34-conj(T34_nf))/(2i)+(T43-conj(T43_nf))/(2i)); 
        end
        
        % fprintf('%d/%d\n', idx_d, Nd); 
    end
    
    % Post-processing 
    rho21               = sum(rho21_tmp .* vx_probs); 
    drho21_dE           = vx_probs * drho21_dE_tmp; % [1, Nf] 
    rho21_ztemp         = rho21_tmp(Nd_mid); 
    drho21_dE_ztemp     = drho21_dE_tmp(Nd_mid, :); 



    alpha_0             = -(kp*RAQR_config.N0*(mu_12^2))/(epsilon_0*hbar*Omega_p)*imag(rho21_ztemp); 
    P_probe             = 29.8e-6; 
    probeTransmission_zt   = exp(-2*alpha_0*RAQR_config.d); 
    Iph0                = (RAQR_config.eta*P_probe)/(hbar*(c0*kp))*e*probeTransmission_zt; 
    gq_PreFactor        = Iph0*(2*kp*RAQR_config.N0*(mu_12^2))/(epsilon_0*hbar*Omega_p); 
    gq_ztemp_arr        = gq_PreFactor * drho21_dE_ztemp;  
    % fprintf('%d/%d: Freq %.1f Hz complete.\n', idx_f, Nf, freq_arr(idx_f)); 
    alpha_0             = -(kp*RAQR_config.N0*(mu_12^2))/(epsilon_0*hbar*Omega_p)*imag(rho21); 
    P_probe             = 29.8e-6; 
    probeTransmission   = exp(-2*alpha_0*RAQR_config.d); 
    Iph0                = (RAQR_config.eta*P_probe)/(hbar*(c0*kp))*e*probeTransmission; 
    gq_PreFactor        = Iph0*(2*kp*RAQR_config.N0*(mu_12^2))/(epsilon_0*hbar*Omega_p); 
    gq_arr              = gq_PreFactor * drho21_dE; 


    gq_struct.gq        = gq_arr; 
    gq_struct.gq_ztemp  = gq_ztemp_arr; 
    gq_struct.pt        = probeTransmission; 
    gq_struct.pt_ztemp  = probeTransmission_zt; 
    gq_struct.rho_21    = rho21; 
    gq_struct.rho21_ztemp = rho21_ztemp; 
    gq_struct.drho21_dE = drho21_dE; 

end

