function [tfI, tfQ] = getPerturbationTransferFunctions(RAQR_config, A_LO)
% This function returns the complex transfer functions from \Omega_sig to
% the change in rho_21. 

    hbar    = 6.626e-34 / (2*pi); 
    a0      = 5.2918e-11; 
    e       = 1.6e-19; 
    mu_MW   = 1443.45*e*a0; 
    % mu_12   = 4.5022*e*a0; 

    % Extract RARQ configurations
    Delta_p = RAQR_config.Delta_p; 
    Delta_c = RAQR_config.Delta_c; 
    Delta_l = RAQR_config.Delta_l; 
    
    
    gamma   = RAQR_config.gamma; 
    gamma_2 = RAQR_config.gamma_2; 
    gamma_3 = RAQR_config.gamma_3; 
    gamma_4 = RAQR_config.gamma_4; 
    gamma_c = RAQR_config.gamma_c; 
    
    Omega_c = RAQR_config.Omega_c; 
    Omega_p = RAQR_config.Omega_p;  
    Omega_RF    = mu_MW*(A_LO)/hbar;        % 1V/m
    
    % Quantum state definition 
    %  |1> 6S1/2    |2> 6P3/2   |3> 47D5/2          |4> 48P3/2 
    Delta_p = -Delta_p; Delta_c = -Delta_c; Delta_l = -Delta_l; 
    
    H = [0,         Omega_p/2,  0,                  0;                  ...
        Omega_p/2,  Delta_p,    Omega_c/2,          0;                  ...
        0,          Omega_c/2,  Delta_p+Delta_c,    Omega_RF/2;         ...
        0,          0,          Omega_RF/2,         Delta_p+Delta_c+Delta_l]; 
    
    Gamma = diag([gamma, gamma+gamma_2, gamma+gamma_3+gamma_c, gamma+gamma_4]); 
    
    
    % Re-order the elements of ρ to construct a vector differential equation 
    % dx/dt = Ax + b
    Aorder = [  1, 1; ...
                2, 1; ...
                3, 1; ...
                4, 1; ...
                1, 2; ...
                2, 2; ...
                3, 2; ...
                4, 2; ...
                1, 3; ...
                2, 3; ...
                3, 3; ...
                4, 3; ...
                1, 4; ...
                2, 4; ...
                3, 4; ...
                4, 4; ];  
    
    [Nvar, ~]   = size(Aorder); 
    A           = zeros(Nvar); 
    
    for ii = 1:Nvar
        for jj = 1:Nvar
            m = Aorder(ii, 1); 
            n = Aorder(ii, 2); 
            k = Aorder(jj, 1); 
            l = Aorder(jj, 2); 
            A(ii, jj) = getA(m, n, k, l, H, Gamma, gamma, gamma_2, gamma_3, gamma_4); 
        end
    end
    
    % Get steady-state solution 
    u4 = zeros(1, 16); u4([1, 6, 11, 16]) = 1; 
    
    Aext        = [A; 1e4*u4];  
    [U, S, V]   = svd(Aext); 
    ss          = diag(S); 
    tmp         = (U'*[zeros(16, 1); 1e4]); 
    xvec_steady = V*inv(diag(ss))*tmp(1:16); 
    
    fprintf('Steady-state rho2_1='); 
    disp(xvec_steady(2)); 
    
    % Perform symmetrization on xvec_steady
    rho_steady = reshape(xvec_steady, [4,4]); 
    rho_steady = (rho_steady+rho_steady')/2; 
    xvec_steady = rho_steady(:); 
    
    % You may find some more characteristics about the matrix A. 
%     [PA, DA]    = eig(A);      % Matrix A has one eigenvalue 0. This corresponds to the steady-state. 
%     lambdas     = diag(DA); 
    
    % Impulse in the H(3,4) element 
    
%     A_another = -1i*(kron(eye(4), H) - kron(H.', eye(4))) - (1/2)*(kron(eye(4), Gamma) + kron(Gamma, eye(4))); 
%     A_another(1, 6) = A_another(1, 6) + gamma_2; 
%     A_another(1, 16) = A_another(1, 16) + gamma_4; 
%     A_another(6, 11) = A_another(6, 11) + gamma_3; 
%     
    % compare
    % Check that they are equal 
    % norm(A_another - A)
    
    %% Now we want to determine the impulse response function of T34/T43 to rho2_1. 
    % Indices of A is well-defined now 
    
    Qtmp = eye(16); 
    Qtmp(:,1) = (u4.')/norm(u4); 
    [Q, R] = qr(Qtmp); 
    Q(:,1) = -Q(:, 1); 
    
    % (y = Q'*x) is the variable substituion 
    A_new = Q'*A*Q; 
    assert(norm(A_new(1, :)) < 1e-6); % The first row of A is all-zero. 
    
    % Eqn: dy(2:n)/dt = y(1)*w + A_tilde*y(2:n), where y1(t) is always
    % constant = (1/2).
    w       = A_new(2:end, 1); 
    A_tilde = A_new(2:end, 2:end); 
    
    % [Vtilde, Dtilde] = eig(A_tilde); 
    
    % Let y(2:16) = z
    z_steady = -(A_tilde)\((1/2)*w); 
    
    % Recover x_steady from z_steady 
    y_steady = [(1/2); z_steady]; 
    x_steady = Q*y_steady; 
    
    
    % Evaluate the characteristic polynomial of A_tilde 
    % pA_tilde = poly(A_tilde); 
    
    syms s; 
    tmp = s*eye(15)-A_tilde; 
    pA2 = det(tmp);   

    pB = cell(15);      % Adjoint matrix of 
    parfor ii = 1:15
        for jj = 1:15
            rsel = true([1, 15]); 
            csel = true([1, 15]);
            rsel(ii) = false; csel(jj) = false; 
            tmp2 = tmp(rsel, csel); 
            pB{jj, ii} = (-1)^(ii+jj)*simplify(det(tmp2)); 
        end
        fprintf('%d/15\n', ii); 
    end
    fprintf('Reduced transfer matrix solved.\n'); 

%% Compute the final transfer functions 

[E34, E43] = getActionMatrices(); 

% Compute the equiv. action matrix in A_tilde 
A_tilde_34 = Q'*E34*Q; 
A_tilde_43 = Q'*E43*Q; 

A_tilde_34 = A_tilde_34(2:end, 2:end); 
A_tilde_43 = A_tilde_43(2:end, 2:end); 

% Compute the response functions of H34 and H43 to rho(2,1)
pB = reshape([pB{:}], [15, 15]); 
z_steady = -(A_tilde)\((1/2)*w); 

Hd_zvec_34 = pB*A_tilde_34*z_steady; 
Hd_zvec_43 = pB*A_tilde_43*z_steady; 

Hd_xvec_34 = Q*[0; Hd_zvec_34]; 
Hd_xvec_43 = Q*[0; Hd_zvec_43]; 

Td34 = sym2poly(Hd_xvec_34(2));
Td43 = sym2poly(Hd_xvec_43(2)); 
Hn   = sym2poly(pA2);


HdI = (Td34+Td43); 
HdQ = (1i)*(Td34-Td43);     % This does not contribute to the probe transmission. 

% T_DCgain = polyval(HdI/(2i), 0)/polyval(Hn, 0); 

tfI = tf(HdI, Hn); % This version does not divide by 2. 
tfQ = tf(HdQ, Hn); 


end

function A = getA(m, n, k, l, H, Gamma, gamma, gamma_2, gamma_3, gamma_4)
    A = 0; 

    % -i (Hρ) term
    A = A + (-1i)*(n==l)*H(m,k); 
    
    % i(ρH) term
    A = A + (1i)*(m==k)*H(l,n); 

    % -(1/2)Gamma*ρ term
    A = A - (1/2)*(n==l)*Gamma(m,k); 

    % -(1/2)ρ*Gamma term
    A = A - (1/2)*(m==k)*Gamma(l,n); 

    if k == 2 && l == 2 && m == 1 && n == 1
        A = A + gamma_2;
    end

    if k == 4 && l == 4 && m == 1 && n == 1
        A = A + gamma_4; 
    end

    if k == 3 && l == 3 && m == 2 && n == 2
        A = A + gamma_3; 
    end

end

