% ======================================================= % 
% 
%  This file scans key parameters to study how the bandwidth changes   
%  
%  w.r.t \Omega_p and \Omega_c.  The next step is to design some optimizer
%  
%  to numerically compute what the optimal \Omega_p and \Omega_c is. 
% ======================================================= % 


clear; clc; close all; 

RAQR_config = configureRAQR(); 

% define scan parameters 
Nc = 20; Np = 21; 
Omega_cArr  = linspace(2*pi*1e6, 2*pi*4e6, Nc); 
Omega_pArr  = linspace(2*pi*1e6, 2*pi*10e6, Np); 
[Omega_cMat, Omega_pMat] = meshgrid(Omega_cArr, Omega_pArr); 

tfs         = struct(); 

useSavedFile    = true; 
fileName        = 'data/tfmat2.mat'; 

if useSavedFile
    data = load(fileName);
    if isfield(data, 'Omega_cMat')
        Omega_cMat = data.Omega_cMat; 
        Omega_pMat = data.Omega_pMat; 
        Omega_cArr = Omega_cMat(1,:); 
        Omega_pArr = Omega_pMat(:,1).'; 
        Nc = length(Omega_cArr); 
        Np = length(Omega_pArr); 
    end
    tfs = data.tfs; 

else 
    for cidx = 1:Nc
        for pidx = 1:Np
            
            cfg         = RAQR_config; 
            cfg.Omega_c = Omega_cMat(pidx, cidx); 
            cfg.Omega_p = Omega_pMat(pidx, cidx); 
            
            [gI1, gQ1, gI2, gQ2]            = getQuantumTransConductanceTFs(cfg); 
            tfs(cidx, pidx).gI1             = gI1; 
            tfs(cidx, pidx).gQ1             = gQ1; 
            tfs(cidx, pidx).gI2             = gI2; 
            tfs(cidx, pidx).gQ2             = gQ2; 
            tfs(cidx, pidx).RAQR_config     = cfg; 
    
        end
    end

    save('data/tfmat.mat', "tfs"); 
end


%% Visualization 

gqs     = zeros(Np, Nc);
BWs     = zeros(Np, Nc); 
f_arr   = logspace(2, 7, 1e4).'; 


for ic = 1:Nc
    for ip = 1:Np
        b = tfs(ic,ip).gI2.b; 
        a = tfs(ic,ip).gI2.a; 
        
        gqs(ip, ic) = abs(dcgain(tf(b,a))); 
        gq_tmpArr   = freqs(b,a,2*pi*f_arr); 
        BWs(ip, ic) = f_arr(find(abs(gq_tmpArr) < gqs(ip, ic)*db2mag(-3), 1, "first")); 
    end
end


set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');


fig1 = figure(1); 
fig1.Position = [679,534,560,420]; 
colormap('jet'); 
contourf(Omega_cMat/(2*pi*1e6), Omega_pMat/(2*pi*1e6), (gqs*1e3), 'ShowText', true, 'LabelFormat', "%0.1f mS", 'FaceAlpha', 0.9); 
% contourf(Omega_pMat/(2*pi*1e6), Omega_cMat/(2*pi*1e6), gqs*1e3); 
colorbar; 
xlabel('\Omega_c/(2\pi) (MHz)'); 
ylabel('\Omega_p/(2\pi) (MHz)')
zlabel('g_q (mS)'); 
view(0, 90); shading interp; 
title('Quantum transconductance'); 



fig2 = figure(2); 
fig2.Position = [1246,529,560,420]; 
colormap('jet'); 
contourf(Omega_cMat/(2*pi*1e6), Omega_pMat/(2*pi*1e6), (BWs/1e3), 'ShowText', true, 'LabelFormat', "%0.0f kHz", 'FaceAlpha', 0.9); 
% contourf(Omega_pMat/(2*pi*1e6), Omega_cMat/(2*pi*1e6), BWs/1e3); 
colorbar;
xlabel('\Omega_c/(2\pi) (MHz)'); 
ylabel('\Omega_p/(2\pi) (MHz)')
zlabel('BW (kHz)'); 
view(0, 90); shading interp; 
title('Bandwidth'); 



fig3    = figure(3);
idx_p   = 20; 
idx_c   = 27; 
fig3.Position = [683,57,560,420]; 
b       = tfs(idx_c, idx_p).gI2.b; 
a       = tfs(idx_c, idx_p).gI2.a; 
gq_arr  = freqs(b, a, 2*pi*f_arr); 

yyaxis left; 
plot(f_arr, mag2db(abs(gq_arr)*1e3)); 
xlabel('Freq (Hz)'); 
ylabel('g_q (dBmS)'); 
Omega_p_MHz = Omega_pArr(idx_p)/(2*pi*1e6); 
Omega_c_MHz = Omega_cArr(idx_c)/(2*pi*1e6); 
fprintf('Omega_p = %.2f MHz, Omega_c = %.2f MHz\n', Omega_p_MHz, Omega_c_MHz); 

yyaxis right; 
plot(f_arr, rad2deg(unwrap(angle(gq_arr)))); 
ylabel('Phase (deg)'); 
set(gca, 'xscale', 'log');
grid on; 


fig4 = figure(4);
fig4.Position = [1247,55,560,420]; 
subplot(2, 1, 1); 
plot(1:Np, Omega_pArr/(2*pi*1e6)); 
ylabel('\Omega_p/(2\pi) (MHz)'); 
subplot(2, 1, 2); 
plot(1:Nc, Omega_cArr/(2*pi*1e6)); 
ylabel('\Omega_c/(2\pi) (MHz)'); 
xlabel('Index'); 


fig5    = figure(5); 
SNRs    = (gqs*1e3)/1.1;  % The SNR calculation formula should be updated. 
C       = (BWs).*log2(1+SNRs); 
contourf(Omega_cMat/(2*pi*1e6), Omega_pMat/(2*pi*1e6), C/1e6, 'ShowText', true, 'LabelFormat', "%0.1f Mbps", 'FaceAlpha', 0.9); 
colorbar;
xlabel('\Omega_c/(2\pi) (MHz)'); 
ylabel('\Omega_p/(2\pi) (MHz)')
zlabel('Rate (bps)'); 
view(0, 90); shading interp;
hold on; 
scatter3(Omega_c_MHz, Omega_p_MHz, C(idx_p, idx_c)/1e6, 100, 'Marker', 'pentagram'); 
title('Rate'); 
legend({'Capacity', 'Operating point'}, 'Location', 'best'); 


% Zero-pole graph
fig6 = figure(6);
zs = roots(b)/(2*pi*1e6); 
ps = roots(a)/(2*pi*1e6); 
scatter(real(zs), imag(zs), 'Marker', 'o'); hold on; 
scatter(real(ps), imag(ps), 'Marker', 'x'); 
xlabel('Real part (MHz)');
ylabel('Imag part (MHz)'); 
title('s-plane'); 
axis equal; grid on; 


%% Utils

