% ================================================================== % 
% 
%   In this file, we plot the coherence factor w.r.t. electrical aperture
%   
%   for Rydberg atomic receivers. 
% 
% ================================================================== % 
clear; clc; close all; 

global fc; 
fc      = 6.9458e9;
c       = physconst('LightSpeed'); 
lambda  = c/fc; 

Ae      = 3*lambda^2/(8*pi); 
ell     = 0.01*lambda; 

Pi_dBm  = -110; 

Pi  = db2pow(Pi_dBm - 30);
eta = 120*pi; 
Ei  = sqrt(2*eta*Pi/Ae); 

% Noise field analysis
T   = 300; 
kB  = physconst('Boltzmann'); 

% pressure p = n*kB*T
% 
h           = 6.626e-34; 

GT_dB       = 10; 

Pt_dBm  = -50; 
Pt      = db2pow(Pt_dBm -30);       % Transmit power -50dBm 
R       = 5; 
SR      = Pt/(4*pi*R^2);        

ER      = sqrt(SR*(2*120*pi));      % Recv electric field peak strength, in V/m
Pr      = SR*Ae; 

Pr_dBm  = pow2db(Pr*1000); 

%% Thermal noise
u_nu = (4*pi/c)*(2*h*fc^3)/(c^2)/expm1(h*fc/(kB*T)); 

% compared to that of an electric field energy
% we = (1/4) \epsilon |E|^2
BW              = 10e6;
epsilon_0       = 8.85e-12; 
Enoise_peak     = sqrt(4*(u_nu*(BW)/2/3)/epsilon_0);        % peak value of the noise field

Enoise_peak_normalized = sqrt(4*(u_nu/2/3)/epsilon_0)*(1e6); % peak value of noise, in μV/m/sqrt(Hz) 

% Explanation of the weird unit nV/cm/sqrt(Hz) ? 
% This means 1.74nV/cm E-field strength can be reliably detected within 1 second of measurement time. 

% Method 2
zeta    = 1; 

spectralRadiance    = 2*fc^2/c^2*kB*T; 
Rn                  = (8*pi)/3*getPhysicalConstant("VacuumWaveImpedance")*spectralRadiance; 
sensitivityLimit    = sqrt(zeta*Rn/2); 

fprintf('Sensitivity limit = %.3f nV/cm/sqrt(Hz)\n', sensitivityLimit*(1e9)/(1e2)); 


%% Simulation 


N_limit = 20; 
    
fc = 6.9e9;  
lambda  = c/fc; 
L_arr   = 0:lambda/20:N_limit*lambda; 

NL      = length(L_arr); 
zeta   = zeros(1, NL); 

spectralRadiance    = 2*fc^2/c^2*kB*T; 
Rn                  = (8*pi)/3*getPhysicalConstant("VacuumWaveImpedance")*spectralRadiance; 
for idx_l = 1:NL
    L           = L_arr(idx_l); 
    fun         = @(x,y)(getZDirectionCorrelationFactors(x-y)); 


    zeta(idx_l)         = (1/(Rn*L^2))*integral2(fun, 0, L, 0, L, 'RelTol', 0.00001); 
end
zeta(1) = 1; 

Rarr = zeros(1, NL); 
Rval = getZDirectionCorrelationFactors(L_arr); 

%% Visualization 
close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');


fig1 = figure(1); 
yyaxis left; 
plot(0:1/20:N_limit, pow2db(zeta)); hold on; 
xlabel('L/\lambda'); ylabel('\zeta(L/\lambda) (dB)'); 

yyaxis right; 
plot(0:1/20:N_limit, Rval/Rn); 
ylabel('$R_{\rm noise}(L)/R_{\rm noise}(0)$', 'Interpreter', 'latex'); 


exportgraphics(fig1, "results/zeta.pdf", "ContentType", "vector"); 
fprintf('Export complete.\n'); 



%% Util Functions 
function Rval = getZDirectionCorrelationFactors(u)
global fc; 
    % fc = 6.9458e9;
    c0 = getPhysicalConstant("LightSpeed"); 
    
    % assume u to be a row vector 
    k0 = 2*pi*fc/c0; 
    
    [m,n] = size(u); 
    Rval = zeros(size(u)); 
    for mm = 1:m
        for nn = 1:n
            if u(mm, nn) == 0
                Rval(mm, nn) = (1/8)*(8/3); 

            else
                [ret, ~]        = get_R(k0*[u(mm, nn), 0, 0].'); 
                Rval(mm, nn)    = ret(3, 3);
            end
        end
    end

    T                   = 300; 
    spectralRadiance    = 2*fc^2/c0^2*getPhysicalConstant("Boltzmann")*T; 
    Rval                = (8*Rval)*pi*getPhysicalConstant("VacuumWaveImpedance")*spectralRadiance; 

end


function [A,x] = Gauss1(N)
    i=N+1;
    f=((sym('t'))^2-1)^i;
    f=diff(f,i);
    t=solve(f);
    for j=1:i
        for k=1:i
            X(j,k)=t(k)^(j-1);
        end
        if mod(j,2)==0
            B(j)=0;
        else
            B(j)=2/j;
        end
    end
    X=inv(X);
    for j=1:i
        A(j)=0;
        x(j)=0;
        for k=1:i
            A(j)=A(j)+X(j,k)*B(k);
            x(j)=x(j)+t(j);
        end
        x(j)=x(j)/k;
    end
end

function g = GaussLegendre(a, b, n, ifunc)
    [A,x]=Gauss1(n);
    g=0;
    for i=1:n+1
        y(i)=(b-a)/2*x(i)+(a+b)/2;
        f(i)=ifunc(y(i));
        g=g+(b-a)/2*f(i)*A(i);
    end
end

