clear; close all; clc; 

RAQR_config.d   = 2e-2;                 % 2cm
RAQR_config.N0  = (4.89e10)*(1e6);      % number of atoms per m^3 

RAQR_config.Delta_p = 2*pi*1e3; 
RAQR_config.Delta_c = 2*pi*0e6;         % Coupling laser detuning
RAQR_config.Delta_l = 2*pi*0e3; 

RAQR_config.gamma   = 0;
RAQR_config.gamma_2 = 2*pi*5.2e6; 
RAQR_config.gamma_3 = 2*pi*3.9e3; 
RAQR_config.gamma_4 = 2*pi*1.7e3; 
RAQR_config.gamma_c = 0; 

% RAQR_config.Omega_c     = 2*pi*0.97e6;                % coupling beam 17.0mW  @ 510nm
RAQR_config.Omega_p     = 2*pi*8.08e6;                  % coupling/control beam 17.0mW  @ 510nm, if this is stronger, you will get a better EIT spectrum 
RAQR_config.Omega_c     = 2*pi*2.05e6;                  % probe    beam 29.8uW  @ 852nm


RAQR_config.lambda_c    = 510e-9; 
RAQR_config.lambda_p    = 852e-9; 


%% Simulation of Probe transmission splitting
sim_config.dt       = 1e-9;
sim_config.T        = 300e-6;
sim_config.t_arr    = 0:sim_config.dt:sim_config.T; 
sim_config.Nt       = length(sim_config.t_arr);

RF_strengths = db2mag(-100:0.2:-20);       % dB(V/m)
fDelta_p_arr = (-50:0.02:50).'*(1e6); 


probeSpectra = getProbeTransmission(RAQR_config, sim_config, fDelta_p_arr, RF_strengths); 
save('probeSpectra.mat', "probeSpectra", "RF_strengths", "sim_config", "fDelta_p_arr", "RAQR_config"); 

%% Visualization
close all; clc; 

saveFigs = false; 
load probeSpectra.mat; 

set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',    12);
set(0,'DefaultAxesFontSize',    14);
set(0,'DefaultLineLineWidth',   1.5);
set(0,'defaultfigurecolor',     'w');

fig1 = figure(1); 
fig1.Position = [100, 100, 600, 400]; 

RFselected  = db2mag(-100:10:-20); 
Nrf_sel     = length(RFselected); 
rf_sel      = zeros(1, Nrf_sel); 

for ii = 1:Nrf_sel
    rf_sel(ii)  = find(RF_strengths == RFselected(ii)); 
end

mycmap = othercolor('RdYlBu_11b', Nrf_sel); 


for idx = 1:Nrf_sel
    plot(fDelta_p_arr/1e6, db2pow(probeSpectra(:, rf_sel(idx))), 'color', mycmap(idx, :)); hold on; 
end

xlabel('Probe detuning (MHz)'); 
ylabel('Probe transmission');


label_cell = cell(Nrf_sel, 1);
a0      = 5.2918e-11; 
e       = 1.6e-19; 
mu_MW   = 1443.45*e*a0; 
hbar    = 6.626e-34 / (2*pi); 

for idx = 1:Nrf_sel
    label_cell{idx} = sprintf('E-field %.2f dBV/m', mag2db(RF_strengths(rf_sel(idx)))); 

    ATsplit = (RAQR_config.lambda_c/RAQR_config.lambda_p)*(mu_MW*RF_strengths(rf_sel(idx))/hbar)/(2*pi); 
    fprintf('ATsplit value %.2f MHz\n', ATsplit/1e6); 
end
legend(label_cell); 
if saveFigs
    exportgraphics(fig1, "results/EIT-AT.pdf", "ContentType", "vector"); 
end

% Plot the input-output relation of the Rydberg atomic receiver.  
fig2 = figure(2); 
% fig2.Position = []; 

zeroDetuningIdx = find(fDelta_p_arr == 0); 
probeRespArr = probeSpectra(zeroDetuningIdx, :); 

pt = db2pow(probeRespArr); 
yyaxis left; 
plot(RF_strengths, pt); 
% set(gca, 'xscale', 'log');
xlabel('MW E-field (V/m)'); ylabel('Probe transmission'); 

yyaxis right; 
plot(RF_strengths(2:end), diff(pt)./diff(RF_strengths)); 
ylabel('Slope ((V/m)^-^1)'); 

if saveFigs
    exportgraphics(fig2, "results/pt_wrt_Efield.pdf", "ContentType", "vector"); 
    fprintf('ProbeTransmission wrt E-field pdf saved.\n'); 
end


%% utils
function probeSpectra = getProbeTransmission(RAQR_config, sim_config, fDelta_p_arr, RF_strengths)
    
    sim_config.dt       = 1e-9;
    sim_config.T        = 50e-6;
    sim_config.t_arr    = 0:sim_config.dt:sim_config.T; 
    sim_config.Nt       = length(sim_config.t_arr);
    
    Nscan               = length(fDelta_p_arr); 
    probeSpectra        = zeros(Nscan, length(RF_strengths)); 
    
    % E_RF                = 0.001*sin(10e3*2*pi*sim_config.t_arr);   
    Nrf = length(RF_strengths); 

    parfor idx_rf = 1:Nrf
        E_RF = RF_strengths(idx_rf); 
        i_probeTransmission_arr = zeros(Nscan, 1); 

        for idx_scan = 1:Nscan
            fDelta_p                            = fDelta_p_arr(idx_scan); 
        
            temp_RAQR_config                    = RAQR_config; 
            temp_RAQR_config.Delta_p            = 2*pi*fDelta_p;  
            % [~, probeResponse]              = transientRAQRSimulation(temp_RAQR_config, sim_config, E_RF); 
            % i_probeTransmission_arr(idx_scan) = probeResponse(end); 
            [~, probeResponseSteady]            = getSteadySolution(temp_RAQR_config, E_RF); 
            i_probeTransmission_arr(idx_scan)   = probeResponseSteady; 
            % fprintf('Scan probe detuning = %.2f MHz complete, probe transmission = %.2f dB\n', fDelta_p/1e6, probeResponse(end)); 
        end

        probeSpectra(:, idx_rf) = i_probeTransmission_arr; 
        fprintf('Scan RF strength %.2f dB(V/m) complete.\n', mag2db(RF_strengths(idx_rf))); 
    end

end


