% =============================================== % 
% 
%  This file plots the frequency response of 
%  
%  the "quantum transconductances".   
% 
% =============================================== % 

clear; clc; close all; 


EAmp_LO         = 0.04;                         % LO field peak value           (V/m) 
EAmp_Sig        = 0.0001;                       % Signal field peak value Ep    (V/m) 
f_IF            = 250e3;                        % Intermediate frequency of 200kHz
Ts              = 10e-6;                        % Symbol period is 10us (100kHz symbol rate)

useNew = true;  % use new functions to evaluate quantum transconductances  
dataPath = './data/'; 

if useNew
    % New configuration
    RAQR_config             = configureRAQR(); 
    hashString              = hashStruct(RAQR_config); 
    [gI1, gQ1, gI2, gQ2, success] = findTransferFunctionFile(hashString, dataPath); 
    
    if ~success
        [gI1, gQ1, gI2, gQ2]    = getQuantumTransConductanceTFs(RAQR_config); % This function is the slowest. 
        save(fullfile(dataPath, 'tfs.mat'), 'hashString', 'gI1', 'gI2', 'gQ1', 'gQ2'); 

    else
        % do nothing 
    end
    gq                      = tf(gI2.b, gI2.a);  % do not save tf structure into .mat files.

else
    % Old configuration 
    load tfs2.mat; 
    GI1 = tf(real(tfI.Numerator{1}), tfI.Denominator{1}); 
    GI2 = tf(imag(tfI.Numerator{1}), tfI.Denominator{1}); 
    GQ1 = tf(real(tfQ.Numerator{1}), tfQ.Denominator{1}); 
    GQ2 = tf(imag(tfQ.Numerator{1}), tfQ.Denominator{1}); 
    dcGainMat = [dcgain(GI1), dcgain(GQ1); dcgain(GI2), dcgain(GQ2)];
    fprintf('DC Gain matrix = \n'); 
    disp(dcGainMat); 
    
    
    % Evaluate the "quantum transconductance" 
    kp          = 2*pi/(852e-9); 
    N0          = (4.89e10)*(1e6);
    epsilon_0   = 8.85e-12;
    a0          = 5.2918e-11; 
    e           = 1.6e-19; 
    mu_MW       = 1443.45*e*a0; 
    mu_12       = 4.5022*e*a0; 
    hbar        = 6.626e-34 / (2*pi); 
    c0          = physconst('LightSpeed');
    Omega_p     = 2*pi*8.08e6;
    
    eta         = 0.8; 
    L           = 2e-2;
    
    load("minors.mat", "x_steady"); 
    alpha_0 = -(kp*N0*(mu_12^2))/(epsilon_0*hbar*Omega_p)*imag(x_steady(2,1));
    P_probe = 29.8e-6; 
    probeTransmission = exp(-2*alpha_0*L); 
    
    Iph0    = (eta*P_probe)/(hbar*(c0*kp))*e*probeTransmission; 
    
    % Model this a a transfer function 
    b   = 2*kp*N0*(mu_12)^2/(epsilon_0*hbar*Omega_p) * Iph0 * (mu_MW/2/hbar)*(GI2.Numerator{1}); 
    a   = GI2.Denominator{1}; 
    gq  = tf(b, a);
    
    % Change in transmission w.r.t. change in input E-field. 
    probeSensitivity = probeTransmission * 2*kp*N0*(mu_12)^2/(epsilon_0*hbar*Omega_p)*(mu_MW)/(2*hbar)*dcgain(GI2); 
    
    fprintf('Est dProbeTransmission/dEsig (V/m)^{-1} = %.3f\n', (abs(probeSensitivity*L))); 
end


%%  Visualization 

close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');
saveFigs = false; 

if useNew
    a = gI2.a; 
    b = gI2.b; 
    GI1 = tf(gI1.b, gI1.a); 
    GQ1 = tf(gQ1.b, gQ1.a);
    GI2 = tf(gI2.b, gI2.a); 
    GQ2 = tf(gQ2.b, gQ2.a); 
end

f_arr   = logspace(2, 8, 500); 
h_arr   = freqs(b, a, 2*pi*f_arr); 


% Print the poles and zeros 
fprintf('Denominator p(x) roots: ');
ps = roots(a)/(2*pi*1e6); 
disp(ps); 

fprintf('Numerator q(x) roots: '); 
zs = roots(b)/(2*pi*1e6); 
disp(zs); 

% The Pole-Zero map 
fig1    = figure(1); 
scatter(real(ps), imag(ps), 'Marker', 'x');  hold on; box on; 
scatter(real(zs), imag(zs), 'Marker', 'o'); 
legend({'Poles', 'Zeros'}); 
axis equal; 
xlabel('Real part (MHz)'); 
ylabel('Imag part (MHz)'); 


% Impulse response and step-response of gq
fig2 = figure(2); 
ts = linspace(0, 7e-6, 1000); 
impl_qg = impulse(gq, ts); 
step_qg = step(gq, ts); 

yyaxis left; 
plot(ts*1e6, impl_qg);  hold on; 
xlabel('time (us)');
ylabel('Impulse response of g_I_2(i\omega)'); 

yyaxis right; 
plot(ts*1e6, step_qg); 
ylabel('Step response of g_I_2(i\omega)'); 
xlim([0, 7]); grid on; 

if saveFigs
    exportgraphics(fig2, "results/imp_step_response.pdf", 'ContentType', 'vector'); 
    fprintf('Imp and step response exported.\n'); 
end


% Four transfer functions 
if useNew
    GainAdj = 1;
else
    GainAdj = 2*kp*N0*(mu_12)^2/(epsilon_0*hbar*Omega_p) * Iph0 * (mu_MW/2/hbar); 
end


fig3 = figure(3); 
fig3.Position = [100, 100, 1200, 800]; 
subplot(2, 2, 1); 
plotAmplitudePhaseResponse(GI1, f_arr, GainAdj); 
xlabel('IF Freq (Hz)'); 
ylabel('g_I_1 (dBmS)'); 

subplot(2, 2, 2); 
plotAmplitudePhaseResponse(GQ1, f_arr, GainAdj); 
xlabel('IF Freq (Hz)'); 
ylabel('g_Q_1 (dBmS)'); 

subplot(2, 2, 3); 
plotAmplitudePhaseResponse(GI2, f_arr, GainAdj); 
xlabel('IF Freq (Hz)'); 
ylabel('g_I_2 (dBmS)'); 

subplot(2, 2, 4); 
plotAmplitudePhaseResponse(GQ2, f_arr, GainAdj); 
xlabel('IF Freq (Hz)'); 
ylabel('g_Q_2 (dBmS)'); 

if saveFigs
    exportgraphics(fig3, "results/quantum_tcond.pdf", "ContentType", "vector"); 
    fprintf('fig3 exported.\n'); 
end

% gI2, quantum transconductance 
fig4 = figure(4); 
plotAmplitudePhaseResponse(GI2, f_arr, GainAdj); 
xlabel('IF Freq (Hz)'); 
ylabel('g_q (dBmS)'); 

if saveFigs
    exportgraphics(fig4, "results/gq.pdf", "ContentType", "vector"); 
    fprintf('fig4 exported.\n'); 
end

% Fig 5 for PPT: Impulse response 
fig5 = figure(5); 
ts = linspace(0, 7e-6, 1000); 
impl_qg = impulse(gq, ts); 

plot(ts*1e6, impl_qg);  hold on; 
xlabel('time (us)');
ylabel('Impulse response of g_q(\tau)'); 
xlim([0, 7]); grid on; 


%% Utils

function plotAmplitudePhaseResponse(sys, f_arr, GainAdj)
    b = GainAdj*sys.Numerator{1}; 
    a = sys.Denominator{1}; 

    h_arr   = freqs(b, a, 2*pi*f_arr); 
    yyaxis left; 
    plot(f_arr, mag2db(abs(h_arr)*1e3)); 
    set(gca, 'xscale', 'log'); 
    grid on; 
    ylabel('g_q (dBmS)'); 

    yyaxis right; 
    plot(f_arr, rad2deg(unwrap(angle(h_arr)))); 
    set(gca, 'xscale', 'log'); grid on; 
    ylabel('phase (deg)'); 

    yyaxis left; 

end




