% ================================================== % 
% 
%   This file provides Rydberg-MIMO link-level simulation 
%   
%   Simulation input: None
%   Simulation output:  1) 
%   
% ================================================== % 
clc; clear all; close all; 

% Setup EM parameters for RAQRs.  
mySetting = true; 


freq    = 6.9e9;
c0      = physconst("lightspeed");

% figure; 
% imagesc(abs(vrx_transferMat)); 
% colorbar; 


% Compute the dipole array response given a planar wave incidence 
Ptx_dBm = 0; 
Gtx_dB  = 5; 

R       = 1000;                         % communication distance 

Srx     = db2pow(Ptx_dBm+Gtx_dB-30)/(4*pi*R^2); 
eta0    = getPhysicalConstant("VacuumWaveImpedance"); 
Erx     = sqrt(2*eta0*Srx);             % peak E-field amplitude in V/m

lambda      = c0/freq; 
kB          = getPhysicalConstant("Boltzmann"); 
T           = 300; 

Ae_dipole   = 3*lambda^2/(8*pi); 
Grx         = Ae_dipole/(lambda^2/(4*pi)); 
Grx_dB      = pow2db(Grx); 
Prx         = Ae_dipole*Srx; 
Zc          = 50; 

Vrx         = sqrt(2*Prx)*sqrt(Zc);     % Waveguide voltage amplitude in [Volts]. 

BW          = 100e3; 
Fc_dB       = 2;
Pn          = kB*T*BW*db2pow(Fc_dB); 

SNRc_dB     = pow2db(Prx/Pn);           % This is, in essence, the power spectral density / noise spectral density. 

% RAQR parameters 
zeta_dB     = -2; 
L           = 0.02; 
% gq          = 1.2e-2;                   % quantum transconductance 
gq          = 1.4e-3; 

% The following (Srx*(2*eta0)/BW/4) is the double-sided PSD of analytical
% Esig, evaluated in the upper sideband. 
PSD_sig     = Srx*(2*eta0)/BW/4 * (L*gq)^2;                                     % in [A^2/Hz] 
PSD_bbr     = ( (4*pi/3)*eta0*(2*kB*T/lambda^2)*db2pow(zeta_dB) ) * (L*gq)^2;   % in [A^2/Hz] 

SNRq_dB     = pow2db( PSD_sig / PSD_bbr);     % SNR before Rydberg - Transimpedance amplifier 
RydGain     = SNRq_dB - SNRc_dB; 

% Transimpedance amplifier (TIA) 
% TIA PARAMETERS (TIA model: DHPCA-100 Variable Gain High-speed Current Amplifier)
TIA_RT      = 10e3;                     % Transimpedance gain                   [V/A] 
TIA_Ipsd    = (1.8e-12/sqrt(2))^2;      % TIA input referred current noise PSD  [A^2/Hz]
TIA_Vpsd    = (2.8e-9/sqrt(2))^2;       % TIA input referred voltage noise PSD  [V^2/Hz] 
TIA_Zin     = 60; 
TIA_Zout    = 50;       
PD_Rs       = 1000;                      % DC bias resistor of the PD 

Kc          = (PD_Rs)/(PD_Rs+TIA_Zin);


% Transimpedance out PSD and SNR for quantum receivers 
if mySetting
    PSD_th2 = (2*kB*T/PD_Rs*Kc^2 + TIA_Ipsd*Kc^2 + TIA_Vpsd*(1/(TIA_Zin+PD_Rs))^2)*(TIA_RT^2);     % thermal+TIA noise [V^2/Hz]
else
    PSD_th2 = (TIA_Vpsd*(1/(TIA_Zin+PD_Rs))^2) * (TIA_RT^2);
end

PSD_bbr2    = PSD_bbr*(Kc*TIA_RT)^2; 
PSD_sig2    = PSD_sig*(Kc*TIA_RT)^2; 

if mySetting
    SNRq2_dB = pow2db(PSD_sig2/(PSD_bbr2+PSD_th2)); 
else
    SNRq2_dB = pow2db(PSD_sig2/PSD_th2);        % Only consider the thermal noise 
end

RydGain2    = SNRq2_dB - SNRc_dB; 

fprintf('Classical Rx SNR = %.3f dB\n', SNRc_dB); 



%% Compare Rydberg-MIMO with classical-MIMO, in terms of spectral efficiency 

gq_seq  = [db2mag(0), db2mag(10), db2mag(20)]*(1.4e-3); 
N_gq    = length(gq_seq); 

% Run the following simulation code for different values of gq. 
PTx_dBm_scanRange = (-10:2:10).'; 
Nscan   = length(PTx_dBm_scanRange); 
Rq      = zeros(Nscan, 2*N_gq); 
Rc      = zeros(Nscan, 2*N_gq);
Rc_noMC = zeros(Nscan, 2);

Ntrial  = 1000; 
Vref    = 1e-3; 

% Classical MIMO parameters 
Nt      = 8; 
Nr      = 8; 
mcMat   = getMutualCouplingMatrix(freq, Nr); 

% SNR results 
SNRq        = zeros(Nscan, Nr); 
SNRc        = zeros(Nscan, Nr);
SNRc_noMC   = zeros(Nscan, Nr); 


for idx_gq = 1:N_gq
    rng(0);

    gq      = gq_seq(idx_gq); 
    fprintf('Simulating for quantum transconductance gq = %.3f mS...\n', gq*1e3); 

    % Step1: Establish the baseband equivalent model  
    P_qref  = 1/( 1/(2*Vref)*TIA_RT*Kc*L*gq*sqrt(8*pi*eta0/lambda^2) )^2;
    Enbbr   = sqrt(8*pi/3*eta0*2/lambda^2*(kB*T)); 
    nw_bbr  = (1/sqrt(2))*(1/Vref/sqrt(2)) * ((TIA_RT*Kc)*L*gq*sqrt(db2pow(zeta_dB))*sqrt(2)*Enbbr);   % This is correct now 
    nw_thv  = (1/sqrt(2))*(1/Vref)*(2.8e-9)        *(TIA_RT/(TIA_Zin+PD_Rs)) ;                         % TIA voltage noise
    nw_thi  = (1/sqrt(2))*(1/Vref)*(1.8e-12)       *(Kc*TIA_RT);                                       % TIA current noise 
    nw_thRs = (1/sqrt(2))*(1/Vref)*sqrt(4*kB*T/PD_Rs)*(Kc*TIA_RT);                                     % Serial resistor Johnson-Nyquist noise 
    
    % in comparison. The thermal voltage of a resistor 
    fprintf('Effective Rs thermal voltage = %.2f nV/sqrt(Hz)\n', (TIA_Zin)/(PD_Rs+TIA_Zin)*sqrt(4*kB*T*PD_Rs)*(1e9)); 
    
    % Compute the Rydberg equiv. baseband noise 
    sigmaq2_w = BW*(nw_bbr^2 + nw_thv^2 + nw_thi^2 + nw_thRs^2); 
    % sigmaq2_w = BW*(nw_bbr^2); 
    
    % sigma   = sqrt(sigmaq2_w) 
    % Test the complex baseband SNR value
    fprintf('Rydberg input SNR = %.3f dB\n', SNRq_dB); 
    fprintf('Photodiode output SNR = %.3f dB\n', SNRq2_dB); 
    
    fprintf('Equiv baseband SNR = %.3f dB\n', ...
        mag2db(   sqrt((db2pow(Ptx_dBm+Gtx_dB-30))/P_qref) * (lambda/(4*pi*R)) / sqrt(sigmaq2_w))); 
    % I have got the point now. These formulas are correct. 
    
    % We may scan the Tx power to get spectral efficiency curves w.r.t.
    % available transmit power. 

    for idx_scan = 1:Nscan
        iPtx_dBm    = PTx_dBm_scanRange(idx_scan); 
        Rq_arr      = zeros(Ntrial, 2); 
        Rc_arr      = zeros(Ntrial, 2); 
        Rc_noMCarr  = zeros(Ntrial, 2); 
        SNRq_arr    = zeros(Ntrial, Nr); 
        SNRc_arr    = zeros(Ntrial, Nr); 
        SNRc_noMCarr = zeros(Ntrial, Nr); 

        for idx_t = 1:Ntrial
            h   = lambda/(4*pi*R)*(randn([Nr, Nt])+1i*randn([Nr, Nt]))/sqrt(2);     % Rayleigh fading channel 
            
            % ============= Rydberg MIMO capacity (no MC) =============== 
            % Communication model: y=Hx+w, where x is unit-power vector 
            % per-channel use MI formula: logdet(I+H Cx H'/sigma2_w)
            Hq                  = sqrt( db2pow(iPtx_dBm+Gtx_dB-30)/P_qref )*h;      % Rydberg baseband channel mtx 
            [Uq, Sq, Vq]        = svd(Hq); 
            [qvec, Cq]          = waterFilling(diag(Sq), sigmaq2_w); 
            Rq_arr(idx_t, 1)    = BW*Cq; 
            Rq_arr(idx_t, 2)    = BW*log2(real(det(eye(Nr)+Hq*(eye(Nt)/Nt)*Hq'/sigmaq2_w)));
            SNRq_arr(idx_t, :)  = sort((qvec.*diag(Sq).^2).', 'descend')/sigmaq2_w; 
    
            % ========= Classical MIMO capacity (with MC) =============== 
            Hc                      = sqrt(db2pow(iPtx_dBm+Gtx_dB+Grx_dB-30))*(mcMat*h); 
            sigmac2_w               = kB*T*BW*db2pow(Fc_dB); 
            [Uc, Sc, Vc]            = svd(Hc); 
            [pvec, Cp]              = waterFilling(diag(Sc), sigmac2_w); 
            Rc_arr(idx_t, 1)        = BW*Cp;
            Rc_arr(idx_t, 2)        = BW*log2(real(det(eye(Nr)+Hc*(eye(Nt)/Nt)*Hc'/sigmac2_w))); 
            SNRc_arr(idx_t, :)      = sort((pvec.*diag(Sc).^2).', 'descend')/sigmac2_w; 

            if idx_gq == 1  % Classical receiver does not scan with quantum transconductance 
            % ========= Classical MIMO capacity (without MC) =============== 
            Hc                      = sqrt(db2pow(iPtx_dBm+Gtx_dB+Grx_dB-30))*(h); 
            sigmac2_w               = kB*T*BW*db2pow(Fc_dB); 
            [Uc, Sc, Vc]            = svd(Hc); 
            [pvec, Cp]              = waterFilling(diag(Sc), sigmac2_w); 
            Rc_noMCarr(idx_t, 1)    = BW*Cp;
            Rc_noMCarr(idx_t, 2)    = BW*log2(real(det(eye(Nr)+Hc*(eye(Nt)/Nt)*Hc'/sigmac2_w))); 
            SNRc_noMCarr(idx_t, :)  = sort((pvec.*diag(Sc).^2).', 'descend')/sigmac2_w; 
            end
        end
    
        Rq(idx_scan, 2*(idx_gq-1)+1:2*(idx_gq-1)+2) = mean(Rq_arr, 1); 
        Rc(idx_scan, 2*(idx_gq-1)+1:2*(idx_gq-1)+2) = mean(Rc_arr, 1); 
        
        if idx_gq == 1
            Rc_noMC(idx_scan, :)    = mean(Rc_noMCarr, 1); 
            SNRq(idx_scan, :)       = sqrt(mean(SNRq_arr.^2, 1)); 
            SNRc(idx_scan, :)       = sqrt(mean(SNRc_arr.^2, 1)); 
            SNRc_noMC(idx_scan, :)  = sqrt(mean(SNRc_noMCarr.^2, 1)); 
        end
        
    end
    fprintf('================ Simulation loop %d complete ===================\n\n', idx_gq); 
end


%% Visualizations 
close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');
saveFigs = true; 

fig1 = figure(1); 

% qMIMO, gq+20dB
% plot(PTx_dBm_scanRange, Rq(:,5)/1e6, 'color', 'magenta'); hold on; 
% plot(PTx_dBm_scanRange, Rq(:,6)/1e6, 'color', 'magenta', 'LineStyle', '--'); 

% cMIMO, no MC 
plot(PTx_dBm_scanRange, Rc_noMC(:,1)/1e6, 'Color', [0 192 0]/255, 'LineStyle', '-'); hold on; 
plot(PTx_dBm_scanRange, Rc_noMC(:,2)/1e6, 'Color', [0 192 0]/255, 'LineStyle', '--'); 

% qMIMO, gq+0dB
plot(PTx_dBm_scanRange, Rq(:,1)/1e6, 'color', 'r'); 
plot(PTx_dBm_scanRange, Rq(:,2)/1e6, 'color', 'r', 'LineStyle','--'); 

% cMIMO, with MC
plot(PTx_dBm_scanRange, Rc(:,1)/1e6, 'color', 'b');
plot(PTx_dBm_scanRange, Rc(:,2)/1e6, 'color', 'b', 'LineStyle','--');


xlabel('Tx power (dBm)'); 
ylabel('Capacity (Mbps)'); 
grid on; 
legend({'Electronic Rx-SVD w/o MC', ...
        'Electronic Rx-Eq w/o MC', ...
        'Quantum Rx-SVD', ...
        'Quantum Rx-Eq', ...
        'Electronic Rx-SVD w/ MC', ...
        'Electronic Rx-Eq w/ MC', ...
        }, 'Location','northwest', 'FontSize', 10); 

if saveFigs
    exportgraphics(fig1, "results/q-mimo.pdf", "ContentType", "vector"); 
    fprintf('Fig file saved.\n'); 
end


fig2    = figure(2); 
colors  = othercolor('RdBu5', Nr); 

for idx = 1:Nr
    plot(PTx_dBm_scanRange, pow2db(SNRq(:, idx)), 'Color', colors(idx, :), 'LineStyle', '-'); hold on; 
    plot(PTx_dBm_scanRange, pow2db(SNRc(:, idx)), 'Color', colors(idx, :), 'LineStyle', '-.'); hold on;  
end

grid on; 
xlabel('Tx power (dBm)'); 
ylabel('Rx subchannel SNR (dB)'); 

legend({'Quantum', 'Classical'}, 'Location', 'southeast'); 

fprintf('Script end.\n'); 

%% Utils 
function [pvec, C] = waterFilling(svec, sigma2_w)
    n           = length(svec); 
    floorHeight = sigma2_w./(svec.^2);
    [ascendFloorHeight, ~] = sort(floorHeight,"ascend"); 
    
    accumPower = 0;
    waterLevel = 0; 
    done = false; 
    for idx = 2:n
        accumPowerTmp = accumPower + (idx-1)*(ascendFloorHeight(idx)-ascendFloorHeight(idx-1)); 
        if accumPowerTmp >= 1
            waterLevel = ascendFloorHeight(idx-1)+(1-accumPower)/(idx-1); 
            done = true; 
            break; 
        end
        accumPower = accumPowerTmp; 
    end

    if ~done
        waterLevel = ascendFloorHeight(n)+(1-accumPowerTmp)/n; 
    end

    pvec = zeros(n, 1); 
    for idx = 1:n
        if waterLevel > floorHeight(idx)
            pvec(idx) = waterLevel - floorHeight(idx);
        else
            pvec(idx) = 0;
        end
    end
    
    C = sum(log2(1+pvec.*(svec.^2)/sigma2_w)); % in bits / ch. use. 
end


%% Test dipole pairs 
% close all; 
% 
% % Define frequency range
% freq = linspace(1e9, 3e9, 20);  % Frequency sweep from 1 GHz to 3 GHz
% 
% % Create two dipole antennas
% d1 = dipole('Length', 0.15, 'Width', 0.01);  % around lambda/2 at 1 GHz
% d2 = dipole('Length', 0.15, 'Width', 0.01);
% 
% % Visualize the antenna geometry
% figure;
% show(d1);
% title('Dipole Antenna');
% 
% % Define the antenna elements and positions
% antArray = linearArray('Element', d1, ...
%                         'NumElements', 2, 'ElementSpacing', 100);  
% 
% 
% s = sparameters(antArray, freq, real(impedance(d1, 2e9))); 
% 
% figure; 
% plot(freq, mag2db(reshape(abs(s.Parameters(2, 1, :)), [1, 20])));  hold on; 
% plot(freq, mag2db(reshape(abs(s.Parameters(1, 1, :)), [1, 20])));
% legend({'s21', 's11'}); 


%% Visualization 
% 
% helement = figure;
% show(mydipole)
% axis tight
% 
% 
% hArray = figure;
% show(dipole_array)
% axis tight
% 
% pattern3Dfig = figure;
% pattern(dipole_array,freq)

