% ======================================================= % 
%   
%   This script studies the WMMSE-based optimization for  
%   multi-band quantum receiver. 
%   
%   Author:    Jieao Zhu
%   Date:      4 July 2025 
%   Comments:  qSDMA and qSFDMA methods 
% 
% ======================================================= % 

clear; clc; close all; rng(0);

M = 2;      % Number of bands 
RAQR_config = configureRAQR("DualBand"); 

% Re-configure some key parameters for tuning the system 
RAQR_config.A_LO1       = 0.04; 
RAQR_config.A_LO2       = 0.03;      
RAQR_config.gamma_5     = 2*pi*1.6e3;  % This value should be aligned with the ARC calculation. 

% Some useful physical constants. 
hbar        = 6.626e-34 / (2*pi); 
a0          = 5.2918e-11; 
e           = 1.6e-19; 
mu_RF1      = 1443.45*e*a0; 
mu_RF2      = 1214.6*e*a0; 
epsilon_0   = 8.85e-12;
mu_12       = 4.5022*e*a0; 
c0          = getPhysicalConstant('LightSpeed'); 
eta0        = getPhysicalConstant('VacuumWaveImpedance'); 
kB          = getPhysicalConstant("Boltzmann"); 
N0          = RAQR_config.N0;
kp          = 2*pi/(RAQR_config.lambda_p); 


mu_arr    = [mu_RF1; mu_RF2]; 
gamma_arr = [RAQR_config.gamma_4; RAQR_config.gamma_5]; 
delta_arr = [-RAQR_config.Delta_l1; RAQR_config.Delta_l2]; 
A_LO_arr  = [RAQR_config.A_LO1; RAQR_config.A_LO2]; 
fc_arr    = [6.9e9; 28e9]; 


%% Uplink Rydberg-MIMO Settings 
Nt = 4;             % Number of UE Tx antennas 
Nr = 5;             % Number of Rydberg-BS Rx antennas, usually larger than Nt. 
S = min([Nt, Nr]);  % Max number of available data streams
K  = 3;             % Number of UEs per band

% Setup TIA Electronic parameters 
BW          = 100e3;    % This is the optical-IF cutoff bandwidth. 
Vref        = 1e-3; 
TIA_RT      = 10e3; 
TIA_Zin     = 60; 
TIA_Zout    = 50;       
PD_Rs       = 1000;                      
T           = 300; 

Kc          = (PD_Rs)/(PD_Rs+TIA_Zin); 
zeta        = db2mag(-2); 

C_sig       = (1/2/Vref)*(TIA_RT*Kc)*(RAQR_config.d)*sqrt(8*pi*eta0)*(fc_arr/c0);
nw_thv      = (1/sqrt(2))*(1/Vref)*(2.8e-9)        *(TIA_RT/(TIA_Zin+PD_Rs)) ;                      % TIA voltage noise
nw_thi      = (1/sqrt(2))*(1/Vref)*(1.8e-12)       *(Kc*TIA_RT);                                    % TIA current noise 
nw_thRs     = (1/sqrt(2))*(1/Vref)*sqrt(4*kB*T/PD_Rs)*(Kc*TIA_RT); 
sigma2_e    = BW*(nw_thv^2 + nw_thi^2 + nw_thRs^2); 

Cqms  = cell(M, 1); % Normalized BBR noise covariance matrix  
Cqms_IFdiv = cell(M, 1); 
for mm = 1:M
    alpha_tmp       = exp(-10-rand(1));
    Cqms{mm}        = (4/3)*kB*T*BW*zeta*toeplitz(alpha_tmp.^(0:Nr-1)); 
    Cqms_IFdiv{mm}  = Cqms{mm}/M; 
end

mcMats  = cell(M, 1);
for mm = 1:M
    mcMat{mm}   = getMutualCouplingMatrix(fc_arr(mm), Nr); 
end


%% Simulation Loops 
Nscan       = 8;
SE_arr      = zeros(Nscan, 6); 
unbal_arr   = zeros(Nscan, 1);

Pmax_arr    = 1e-3*logspace(-1,1,Nscan).'; 
Nmc         = 75; 

alpha       = ones(M, K); 
vbs         = 0; 
method      = "Armijo-Goldstein"; 
NiterMax    = 150;


parfor idx_scan = 1:Nscan
    rng(2); 
    
    SE_MCtrials         = zeros(Nmc, 6); 
    unbal_MCtrials      = zeros(Nmc, 1); 
    Pmax                = zeros(M, K);          % in Watts
    
    for idx_mc = 1:Nmc

        % Generate max. power allocation matrix
        for mm = 1:M
            for kk = 1:K
                Pmax(mm, kk) = Pmax_arr(idx_scan)*(1+0.25*(1-2*rand(1))); 
            end
        end
        
        % Generate channel 
        Hmks    = cell(M, K);   % Uplink MU-MIMO Channels
        Hmks_mc = cell(M, K);   
        for mm = 1:M
            for kk = 1:K
                rmk = 500 + 1000*rand(1);  % distance
                Hmks{mm,kk} = (c0/fc_arr(mm))/(4*pi*rmk)*(randn([Nr, Nt])+1i*randn([Nr, Nt]))/sqrt(2); 
                Hmks_mc{mm,kk} = mcMat{mm}*Hmks{mm,kk}; 
            end
        end
        eps         = 1e-4;     % Error tolerance
        
        % Method 1: qWMMSE q-opt enabled 
        lr_ALO      = 6e-6; 
        opt_out     = qWMMSE(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms, C_sig, alpha, Pmax, sigma2_e, NiterMax, eps, lr_ALO, method, vbs); 
        wRate_arr   = opt_out.wRate_trace; 
        gq_trace    = opt_out.gq_trace; 
        Palloc_trace = opt_out.Palloc_trace;
        SE_MCtrials(idx_mc, 1) = wRate_arr(end)/M;              % BW_RF = M*BW_IF

        rr                          = abs(gq_trace(end, 1)/gq_trace(end, 2)); 
        unbal_MCtrials(idx_mc, 1)   = 2/(rr+1/rr);  % [0,1], where 0 means unbalance, 1 means balance. 

        % Method 2: qFDMA; each user group bandwidth = (1/M)*BW; electronic noise = sigma2_e/M;   
        opt_out     = qWMMSE_IFdiv(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms_IFdiv, C_sig, alpha, Pmax, sigma2_e/M, NiterMax, eps, lr_ALO, method, vbs); 
        SE_MCtrials(idx_mc, 2) = opt_out.wRate_trace(end)/M;    % BW_RF = M*(BW_IF/M) 

        % Method 3: qSDMA without qOpt; 
        lr_ALO      = 0;
        opt_out     = qWMMSE(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms, C_sig, alpha, Pmax, sigma2_e, NiterMax, eps, lr_ALO, method, vbs); 
        SE_MCtrials(idx_mc, 3) = opt_out.wRate_trace(end)/M;     % BW_RF = M*BW_IF

        % Method 4: qFDMA without qOpt; 
        lr_ALO      = 0; 
        opt_out     = qWMMSE_IFdiv(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms_IFdiv, C_sig, alpha, Pmax, sigma2_e/M, NiterMax, eps, lr_ALO, method, vbs); 
        SE_MCtrials(idx_mc, 4) = opt_out.wRate_trace(end)/M; 

        % Method 5: cSDMA without mutual coupling; BW = BW_IF
        opt_out = cWMMSE(Hmks, alpha, Pmax, BW*kB*T, NiterMax, eps, vbs); 
        SE_MCtrials(idx_mc, 5) = opt_out.wRate_trace(end);      % BW_RF = BW_IF 

        % Method 6: cSDMA with mutual coupling 
        opt_out = cWMMSE(Hmks_mc, alpha, Pmax, BW*kB*T, NiterMax, eps, vbs); 
        SE_MCtrials(idx_mc, 6) = opt_out.wRate_trace(end);      % BW_RF = BW_IF 

    end

    SE_arr(idx_scan, :)     = mean(SE_MCtrials, 1); 
    unbal_arr(idx_scan)     = mean(unbal_MCtrials); 
    
    fprintf('Scan %d/%d\n', idx_scan, Nscan); 
end

save('opt.mat'); 
fprintf('File saved at opt.mat\n'); 


%% Visualization 
close all; 
load opt.mat; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor',    'w');
saveFigs = true; 

% figure(1); 
% plot((1:NiterMax).', wRate_arr); 
% 
% xlabel('Iterations'); 
% ylabel('Sum Rate (bps/Hz)'); grid on; 

% figure(2); 
% plot((1:NiterMax).', mag2db(1e3*abs(gq_trace(:,1)))); hold on; 
% plot((1:NiterMax).', mag2db(1e3*abs(gq_trace(:,2))));
% xlabel('Iterations'); 
% ylabel('Quantum Transconductance (dBmS)'); grid on;
% legend({'Band-1', 'Band-2'}); 

% figure(3); 
% plot((1:NiterMax).', ALO_trace(:,1)); hold on; 
% plot((1:NiterMax).', ALO_trace(:,2));
% xlabel('Iterations'); 
% ylabel('LO Amplitude (V/m)'); grid on;
% legend({'Band-1', 'Band-2'}); 

% figure(4); 
% plot((1:NiterMax).', abs(gqder_trace(:,1)+1i*gqder_trace(:,2))); hold on; 
% xlabel('Iterations'); 
% ylabel('g_q derivative norm'); 
% 
% figure(5); 
% legends = cell(M*K, 1); 
% for mm = 1:M
%     for kk = 1:K
%         plot((1:NiterMax).', Palloc_trace(:,mm, kk)); hold on; 
%         legends{kk+K*(mm-1)} = sprintf('m=%d, k=%d', mm, kk); 
%     end
% end
% legend(legends); 
% title('Power Allocation'); 

% colors = othercolor('RdYlBu_11b', 6); 
Ncolors = 4; 
colors = interp1((0:255)/256*Ncolors, colormap('turbo'), flip(1/2+(0:(Ncolors-1))), "linear", "extrap" ); 

fig5 = figure(5); 
fig5.Position = [744,530,560,420]; 

plot(pow2db(1e3*Pmax_arr), SE_arr(:,2),     'Color', colors(1,:), 'Marker','o'); hold on; 
plot(pow2db(1e3*Pmax_arr), SE_arr(:,4),     'Color', colors(2,:), 'Marker', 'square'); 
plot(pow2db(1e3*Pmax_arr), SE_arr(:,1),     'Color', colors(3,:), 'Marker','^');  
plot(pow2db(1e3*Pmax_arr), SE_arr(:,3),     'Color', colors(4,:), 'Marker','v');  
xlabel('Tx power (dBm)'); 
ylabel('Spectral efficiency (bps/Hz)'); 
grid on; 
legend({'qFDMA-Opt', 'qFDMA-NoOpt', 'qSDMA-Opt', 'qSDMA-NoOpt'}, 'Location', 'northwest', 'FontSize', 10); 

if saveFigs
    exportgraphics(fig5, 'results/Technical_qOpt.pdf', 'ContentType', 'vector'); 
    fprintf('Technical_qOpt.pdf saved.\n'); 
end

% Fig.~6 Classical v.s. Quantum receivers, Achievable Rate. 
fig6 = figure(6); 
plot(pow2db(1e3*Pmax_arr), SE_arr(:,5)*BW/1e6,      'Color', 'k', 'Marker', 'none', 'LineStyle', '--');  hold on;
plot(pow2db(1e3*Pmax_arr), SE_arr(:,1)*(BW*M)/1e6,  'Color', colors(1,:), 'Marker','^'); 
plot(pow2db(1e3*Pmax_arr), SE_arr(:,6)*BW/1e6,      'Color', colors(3,:), 'Marker', 'diamond'); 
plot(pow2db(1e3*Pmax_arr), SE_arr(:,2)*(BW)/1e6,  'Color', colors(4,:), 'Marker','o');
xlabel('Tx power (dBm)'); 
ylabel('Achievable rate (Mbps)'); 
grid on; 
legend({'cSDMA without MC', 'qSDMA-Opt', 'cSDMA with MC', 'qFDMA-Opt'}, 'Location', 'northwest', 'FontSize', 10); 

if saveFigs
    exportgraphics(fig6, 'results/Technical_CvsQ.pdf', 'ContentType','vector'); 
    fprintf('Technical_CvsQ.pdf saved.\n'); 
end


% Fig.~7 power balance level 
fig7 = figure(7); 
fig7.Position = [1307.4,530.6,560,420]; 
plot(pow2db(1e3*Pmax_arr), unbal_arr); 
xlabel('Tx power (dBm)'); 
ylabel('Balance level'); 



