% ======================================================= % 
%   
%   This script studies the WMMSE-based optimization trace. 
%   
%   Author:    Jieao Zhu
%   Date:      6 July 2025 
%   Comments:  qSDMA and qSFDMA methods 
% 
% ======================================================= % 

clear; clc; close all; rng(0);

M = 2;      % Number of bands 
RAQR_config = configureRAQR("DualBand"); 

% Re-configure some key parameters for tuning the system 
RAQR_config.A_LO1       = 0.04; 
RAQR_config.A_LO2       = 0.03;       
RAQR_config.gamma_5     = 2*pi*1.6e3; 

% Some useful physical constants. 
hbar        = 6.626e-34 / (2*pi); 
a0          = 5.2918e-11; 
e           = 1.6e-19; 
mu_RF1      = 1443.45*e*a0; 
mu_RF2      = mu_RF1; 
epsilon_0   = 8.85e-12;
mu_12       = 4.5022*e*a0; 
c0          = getPhysicalConstant('LightSpeed');
eta0        = getPhysicalConstant('VacuumWaveImpedance'); 
kB          = getPhysicalConstant("Boltzmann"); 
N0          = RAQR_config.N0;
kp          = 2*pi/(RAQR_config.lambda_p); 


mu_arr    = [mu_RF1; mu_RF2]; 
gamma_arr = [RAQR_config.gamma_4; RAQR_config.gamma_5]; 
delta_arr = [-RAQR_config.Delta_l1; RAQR_config.Delta_l2]; 
A_LO_arr  = [RAQR_config.A_LO1; RAQR_config.A_LO2]; 
fc_arr    = [6.9e9; 28e9]; 


%% Uplink Rydberg-MIMO Settings 
Nt = 4;             % Number of UE Tx antennas 
Nr = 5;             % Number of Rydberg-BS Rx antennas, usually larger than Nt. 
S = min([Nt, Nr]);  % Max number of available data streams
K  = 3;             % Number of UEs per band

% Setup TIA Electronic parameters 
BW          = 100e3; 
Vref        = 1e-3; 
TIA_RT      = 10e3; 
TIA_Zin     = 60; 
TIA_Zout    = 50;       
PD_Rs       = 1000;                      
T           = 300; 

Kc          = (PD_Rs)/(PD_Rs+TIA_Zin); 
zeta        = db2mag(-2); 

C_sig       = (1/2/Vref)*(TIA_RT*Kc)*(RAQR_config.d)*sqrt(8*pi*eta0)*(fc_arr/c0);
nw_thv      = (1/sqrt(2))*(1/Vref)*(2.8e-9)        *(TIA_RT/(TIA_Zin+PD_Rs)) ;                      % TIA voltage noise
nw_thi      = (1/sqrt(2))*(1/Vref)*(1.8e-12)       *(Kc*TIA_RT);                                    % TIA current noise 
nw_thRs     = (1/sqrt(2))*(1/Vref)*sqrt(4*kB*T/PD_Rs)*(Kc*TIA_RT); 
sigma2_e    = BW*(nw_thv^2 + nw_thi^2 + nw_thRs^2); 

Cqms  = cell(M, 1); % Normalized BBR noise covariance matrix  
Cqms_IFdiv = cell(M, 1); 
for mm = 1:M
    alpha_tmp       = exp(-10-rand(1));
    Cqms{mm}        = (4/3)*kB*T*BW*zeta*toeplitz(alpha_tmp.^(0:Nr-1)); 
    Cqms_IFdiv{mm}  = Cqms{mm}/M; 
end

mcMats  = cell(M, 1);
for mm = 1:M
    mcMat{mm}   = getMutualCouplingMatrix(fc_arr(mm), Nr); 
end

%% Simulation Loops 
Nscan       = 9;
Nmethod     = 6; 

optOutputs  = cell(Nscan, Nmethod); 

Pmax_arr    = 1e-3*logspace(-1, 1, Nscan);  
Nmc         = 75; 

alpha       = ones(M, K); 
vbs         = 0; 
method      = "Armijo-Goldstein"; 
NiterMax    = 300;



AchRate_MCtrials    = zeros(Nmc, 6); 
unbal_MCtrials      = zeros(Nscan, 1); 
Pmax                = zeros(M, K);  % in Watts
    

for idx_scan = 1:Nscan
    rng(1); 

    % Generate max. power allocation matrix 
    for mm = 1:M
        for kk = 1:K
            Pmax(mm, kk) = Pmax_arr(idx_scan)*(1+0.25*(1-2*rand(1))); 
        end
    end
    
    % Generate channel 
    Hmks    = cell(M, K);   % Uplink MU-MIMO Channels
    Hmks_mc = cell(M, K); 
    for mm = 1:M
        for kk = 1:K
            rmk = 500 + 1000*rand(1);  % distance
            Hmks{mm,kk} = (c0/fc_arr(mm))/(4*pi*rmk)*(randn([Nr, Nt])+1i*randn([Nr, Nt]))/sqrt(2); 
            Hmks_mc{mm,kk} = mcMat{mm}*Hmks{mm,kk}; 
        end
    end
    eps         = 1e-4;     % Error tolerance
    
    % Method 1: qWMMSE q-opt enabled 
    lr_ALO      = 6e-6; 
    opt_out     = qWMMSE(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms, C_sig, alpha, Pmax, sigma2_e, NiterMax, eps, lr_ALO, method, vbs); 
    gq_trace    = opt_out.gq_trace; 
    optOutputs{idx_scan, 1} = opt_out;   

    rr          = abs(gq_trace(end, 1)/gq_trace(end, 2)); 
    unbal_MCtrials(idx_scan, 1)  = 2/(rr+1/rr);  % [0,1], where 0 means unbalance, 1 means balance. 
    

    % Method 2: qSFDMA q-opt enabled 
    opt_out     = qWMMSE_IFdiv(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms_IFdiv, C_sig, alpha, Pmax, sigma2_e/M, NiterMax, eps, lr_ALO, method, vbs); 
    opt_out.wRate_trace = opt_out.wRate_trace/M; 
    optOutputs{idx_scan, 2} = opt_out; 

    % Method 3: qSDMA without qOpt 
    lr_ALO      = 0;
    opt_out     = qWMMSE(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms, C_sig, alpha, Pmax, sigma2_e, NiterMax, eps, lr_ALO, method, vbs); 
    optOutputs{idx_scan, 3} = opt_out; 

    % Method 4: qSFDMA without qOpt
    lr_ALO      = 0; 
    opt_out     = qWMMSE_IFdiv(RAQR_config, mu_arr, gamma_arr, delta_arr, A_LO_arr, Hmks, Cqms_IFdiv, C_sig, alpha, Pmax, sigma2_e/M, NiterMax, eps, lr_ALO, method, vbs); 
    opt_out.wRate_trace = opt_out.wRate_trace/M; 
    optOutputs{idx_scan, 4} = opt_out; 

    % Method 5: cSDMA without mutual coupling 
    optOutputs{idx_scan, 5} = cWMMSE(Hmks, alpha, Pmax, BW*kB*T, NiterMax, eps, vbs); 

    % Method 6: cSDMA with mutual coupling 
    optOutputs{idx_scan, 6} = cWMMSE(Hmks_mc, alpha, Pmax, BW*kB*T, NiterMax, eps, vbs); 
    
    fprintf('Complete: %d/%d\n', idx_scan, Nscan); 
end


    
%% Visualization of opt_out 
close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor',    'w');
saveFigs = true; 

% Figure 1: wRate w.r.t. Iter

fig1 = figure(1); 
plotIndices = [9, 7, 5, 3];
legends = cell(length(plotIndices), 1); 
colors = interp1((0:255)/256*length(plotIndices), colormap('turbo'), flip(1/2+(0:(length(plotIndices)-1))), "linear", "extrap" ); 
for idx = 1:length(plotIndices)
    opt_out     = optOutputs{plotIndices(idx), 1}; 
    realIterMax = length(opt_out.wRate_trace); 
    legends{idx} = sprintf('P_m_a_x=%.1f dBm', pow2db(1e3*Pmax_arr(plotIndices(idx))) ); 
    plot((1:realIterMax).', opt_out.wRate_trace, 'Color', colors(idx, :)); hold on; 

end
xlabel('Iterations'); 
ylabel('Sum Rate (bps/Hz)'); grid on; 
legend(legends, 'Location', 'best', 'FontSize', 10); 
if saveFigs
   exportgraphics(fig1, 'results/Technical_wRate_Iters.pdf', 'ContentType', 'vector'); 
end

% Fig.~2 for gq optimization trace 
plotIndices = [9, 7, 5, 3]; 
fig2 = figure(2); 
legends = cell(length(plotIndices), 1); 
for idx = 1:length(plotIndices)
    opt_out = optOutputs{plotIndices(idx), 1}; 
    
    yyaxis left; 
    plot((1:realIterMax).', mag2db(1e3*abs(opt_out.gq_trace(:,1)))); hold on; 
    ylabel('g_q_1 (dBmS)'); 
    yyaxis right; 
    plot((1:realIterMax).', mag2db(1e3*abs(opt_out.gq_trace(:,2)))); hold on; 
    ylabel('g_q_2 (dBmS)'); 
    xlabel('Iterations'); grid on;

    legends{idx} = sprintf('P_m_a_x=%.1f dBm', pow2db(1e3*Pmax_arr(plotIndices(idx))) ); 

end
yyaxis left; 
legend(legends, 'Location', 'best', 'FontSize', 10); 
if saveFigs
   exportgraphics(fig2, 'results/Technical_gqs_Iters.pdf', 'ContentType', 'vector'); 
   fprintf('Technical_gqs_Iters.pdf saved.\n'); 
end


% Fig. 3
plotIndices = [9, 7, 5, 3]; 
fig3 = figure(3); 
legends = cell(length(plotIndices), 1); 
for idx = 1:length(plotIndices)
    opt_out = optOutputs{plotIndices(idx), 1}; 
    
    yyaxis left; 
    plot((1:realIterMax).', 1e3*opt_out.ALO_trace(:,1)); hold on; 
    ylabel('LO_1 amplitude (mV/m)'); grid on;
    yyaxis right; 
    plot((1:realIterMax).', 1e3*opt_out.ALO_trace(:,2)); hold on; 
    ylabel('LO_2 amplitude (mV/m)'); grid on;

    xlabel('Iterations'); 
    legends{idx} = sprintf('P_m_a_x=%.1f dBm', pow2db(1e3*Pmax_arr(plotIndices(idx))) ); 
end
legend(legends, 'Location', 'best', 'FontSize', 10);
if saveFigs
   exportgraphics(fig3, 'results/Technical_LOAmps_Iters.pdf', 'ContentType', 'vector'); 
   fprintf('Technical_LOAmps_Iters.pdf saved.\n'); 
end


% Fig. 4 for gq derivative 
fig4 = figure(4);
plotIndices = [9, 7, 5, 3]; 
legends = cell(length(plotIndices), 1); 
for idx = 1:length(plotIndices)
    opt_out = optOutputs{plotIndices(idx), 1}; 
    gqdnorm_trace = abs(opt_out.gqder_trace(:,1)+1i*opt_out.gqder_trace(:,2)); 
    legends{idx} = sprintf('P_m_a_x=%.1f dBm', pow2db(1e3*Pmax_arr(plotIndices(idx))) ); 
    plot((1:realIterMax).', gqdnorm_trace);  hold on; 
end
xlabel('Iterations'); grid on; 
ylabel('g_q derivative norm'); 
legend(legends, 'Location', 'best', 'FontSize', 10); 


fig5 = figure(5); 
opt_out = optOutputs{3, 1}; 
legends = cell(M*K, 1); 
for mm = 1:M
    for kk = 1:K
        plot((1:NiterMax).', opt_out.Palloc_trace(:,mm, kk)); hold on; 
        legends{kk+K*(mm-1)} = sprintf('m=%d, k=%d', mm, kk); 
    end
end
legend(legends); 
title('Power Allocation'); 

% colors = othercolor('RdYlBu_11b', 6); 
% colors = interp1((0:255)/256*6, colormap('turbo'), flip(1/2+(0:(6-1))), "linear", "extrap" ); 
% 
% fig5 = figure(5); 
% fig5.Position = [744,530,560,420]; 
% plot(pow2db(1e3*Pmax_arr), AchRate_arr(:,5), 'Color', colors(1,:), 'Marker','>');hold on; 
% plot(pow2db(1e3*Pmax_arr), AchRate_arr(:,6), 'Color', colors(2,:), 'Marker', 'diamond'); 
% plot(pow2db(1e3*Pmax_arr), AchRate_arr(:,1), 'Color', colors(3,:), 'Marker','^'); 
% plot(pow2db(1e3*Pmax_arr), AchRate_arr(:,2), 'Color', colors(4,:), 'Marker','o');
% plot(pow2db(1e3*Pmax_arr), AchRate_arr(:,3), 'Color', colors(5,:), 'Marker','v');
% plot(pow2db(1e3*Pmax_arr), AchRate_arr(:,4), 'Color', colors(6,:), 'Marker', 'square');
% xlabel('Tx power (dBm)'); 
% ylabel('Sum rate (bps/Hz)'); 
% grid on; 
% legend({'cSDMA without MC', 'cSDMA with MC', 'qSDMA-Opt', 'qSFDMA-Opt', 'qSDMA-NoOpt', 'qSFDMA-NoOpt'}, 'Location', 'best'); 
% 
% if saveFigs
%     exportgraphics(fig5, 'results/qOpt.pdf', 'ContentType','vector'); 
% end
% 
% fig6 = figure(6); 
% fig6.Position = [1307.4,530.6,560,420]; 
% plot(pow2db(1e3*Pmax_arr), unbal_arr); 
% xlabel('Tx power (dBm)'); 
% ylabel('Balance level'); 
% 
% 
% 
