% ================================================= % 
% 
%  This script is for writing the response: Quantifying
%  the transit effect. 
% 
%  Author: Jieao Zhu  Date: 11 Oct 2025 
% 
% ================================================= % 

clear; close all; clc; 


%%  Setup quantum system parameters 
RAQR_config_transit = configureRAQR("Transit");     % The transit effect is captured by the \gamma parameter. 

% Re-configure some important parameters 
% RAQR_config_transit.Omega_c     = 2*pi*8e6;         % coupling beam 17.0mW  @ 510nm
% RAQR_config_transit.Omega_p     = 2*pi*2e6;         % probe    beam 29.8uW  @ 852nm

RAQR_config = configureRAQR("Normal"); 
% RAQR_config.Omega_c = 2*pi*8e6; 
% RAQR_config.Omega_p = 2*pi*2e6; 


eps = 1e-6; 
N_freq = 2; 
N_ALO  = 100; 
Temp = 300; 

freq_arr = logspace(1, 5, N_freq); 
A_LO_arr = logspace(-3, 1, N_ALO);  

ptArr   = zeros(N_ALO, 2); % (N_ALO, 2) (1): transit (2): non-transit  
ptArr_a = zeros(N_ALO, 2); 

for idx = 1:N_ALO 
    new_config      = RAQR_config_transit; 
    new_config.A_LO = A_LO_arr(idx); 
    gq_struct       = getNumericalFreqResponse(new_config, Temp, freq_arr); 
    ptArr(idx, 1)   = gq_struct.pt; 

    gq_struct_a     = getAnalyticalFreqResponse(new_config, Temp, freq_arr); 
    ptArr_a(idx, 1) = gq_struct_a.pt;
    
    % Non-transit 
    new_config      = RAQR_config; 
    new_config.A_LO = A_LO_arr(idx); 
    gq_struct       = getNumericalFreqResponse(new_config, Temp, freq_arr); 
    ptArr(idx, 2)   = gq_struct.pt; 
    gq_struct_a     = getAnalyticalFreqResponse(new_config, Temp, freq_arr); 
    ptArr_a(idx, 2) = gq_struct_a.pt; 

    fprintf('%d/%d\n', idx, N_ALO); 
end


%% Visualization 
close all; clc; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',    12);
set(0,'DefaultAxesFontSize',    14);
set(0,'DefaultLineLineWidth',   1.5);
set(0,'defaultfigurecolor',     'w');

savefigs = false; 

fig1 = figure(1); 
fig1.Position = [1137, 311, 560, 420]; 
plot(A_LO_arr.'/1e2, ptArr_a(:,2), 'LineWidth', 2.0, 'LineStyle', '--', 'Color', 'b'); grid on; hold on;  
plot(A_LO_arr.'/1e2, ptArr(:,2), 'LineWidth', 0.5, 'Marker', 'x',       'Color', 'b'); 
plot(A_LO_arr.'/1e2, ptArr_a(:,1), 'LineWidth', 2.0, 'LineStyle', '--', 'Color', 'r'); grid on; hold on; 
plot(A_LO_arr.'/1e2, ptArr(:,1), 'LineWidth', 0.5, 'Marker', 'x',       'Color', 'r'); 
set(gca, 'xscale', 'log'); 
xlabel('LO Amplitude (Vp/cm)'); 
ylabel('Probe transmission'); 
legend({"Analytical (non-transit)", "Numerical (non-transit)", "Analytical (transit)", "Numerical (transit)"}, 'location', 'best'); 


if savefigs
    exportgraphics(fig1, "./results/response/T_PTResponse_transit.pdf", "ContentType", "vector"); 
    % PT: Probe Transmission. 
    fprintf('Figure 2 saved.\n'); 
end
% TODO: We may find the transfer function from gq_arr. 
% We need some algorithm to do this. 
% We have a problem here. Why does pt increase as E_LO increase from 0? 


fprintf('Script ended\n'); 