% ========================================================================================== % 
% 
%    Description: This script generates the probe response (dB) as a function of 
%    the E_LO field strength (V/m). 
%    
%    Feature: All the dephasing rates gamma_{2,3,4} are jointly considered. (T=0K)  
%    [TWC] Fig. 2 
%    
%    Author:  Jieao Zhu
%    Version: June 2025
%   
% 
% ========================================================================================== % 
clear; clc; close all; 

RAQR_config.d   = 2e-2;                     % 2cm
RAQR_config.N0  = (4.89e10)*(1e6);          % number of atoms per m^3 

RAQR_config.Delta_p = 2*pi*0.01e3; 
RAQR_config.Delta_c = 2*pi*0.01e3;          % Coupling laser detuning
RAQR_config.Delta_l = 2*pi*0.015e3; 

RAQR_config.gamma   = 0;
RAQR_config.gamma_2 = 2*pi*5.2e6; 
RAQR_config.gamma_3 = 2*pi*3.9e3/2; 
RAQR_config.gamma_4 = 2*pi*1.7e3/2; 
RAQR_config.gamma_c = 0; 

% RAQR_config.Omega_c     = 2*pi*0.97e6;                % coupling beam 17.0mW  @ 510nm
RAQR_config.Omega_p     = 2*pi*8.08e6;                  % coupling/control beam 17.0mW  @ 510nm, if this is stronger, you will get a better EIT spectrum 
RAQR_config.Omega_c     = 2*pi*2.05e6;                  % probe    beam 29.8uW  @ 852nm


RAQR_config.lambda_c    = 510e-9; 
RAQR_config.lambda_p    = 852e-9; 



%% Simulation 
RF_strengths = db2mag(-120:1:0);  

N_rf = length(RF_strengths); 


% gamma_mat = [   0, 5.2e6,   3.9e3,      1.7e3; ...
%                 0, 2*5.2e6, 3.9e3,      1.7e3; ...
%                 0, 5.2e6,   2*3.9e3,    1.7e3; ...
%                 0, 5.2e6,   3.9e3,      2*1.7e3; ...
%                 0, 5.2e6,   1e2,        1e2]*(2*pi); 

% Simulation batch 1: Varying gamma_4. 
gamma_4_arr = [0.01e3; 0.5e3; 1.0e3; 1.7e3]; 
gamma_mat1 = [zeros(length(gamma_4_arr), 1),  5.2e6*ones(length(gamma_4_arr), 1),  3.9e3*ones(length(gamma_4_arr), 1), gamma_4_arr]*(2*pi); 

% gamma_mat1 = [  0, 5.2e6,   3.9e3,      0.01e3; ...
%                 0, 5.2e6,   3.9e3,      0.5e3; ...
%                 0, 5.2e6,   3.9e3,      1.0e3; ...
%                 0, 5.2e6,   3.9e3,      1.7e3;]*(2*pi); 


[N_g1, ~]    = size(gamma_mat1); 
pT1          = zeros(N_g1, N_rf);

for idx_g = 1:N_g1
    temp_RAQR_config = RAQR_config; 
    temp_RAQR_config.gamma_2 = gamma_mat1(idx_g, 2); 
    temp_RAQR_config.gamma_3 = gamma_mat1(idx_g, 3); 
    temp_RAQR_config.gamma_4 = gamma_mat1(idx_g, 4); 

    for idx = 1:N_rf
        [~, probeResponseSteady] = getSteadySolution(temp_RAQR_config, RF_strengths(idx)); 
        pT1(idx_g, idx) = db2pow(probeResponseSteady); 
    end

end


% Simulation batch 2: varying gamma_3 together with gamma_4. 

% gamma_mat2 = [  0, 5.2e6,   0.01e3,      1.7e3; ...
%                 0, 5.2e6,   0.9e3,      1.7e3; ...
%                 0, 5.2e6,   1.5e3,      1.7e3; ...
%                 0, 5.2e6,   2.7e3,      1.7e3; ...
%                 0, 5.2e6,   3.9e3,      1.7e3]*(2*pi); 
gamma_3_arr = gamma_4_arr/(1.7e3)*(3.9e3); 
gamma_mat2 = [zeros(length(gamma_4_arr), 1),  5.2e6*ones(length(gamma_4_arr), 1),  gamma_3_arr, gamma_4_arr]*(2*pi); 

[N_g2, ~]    = size(gamma_mat2); 
pT2         = zeros(N_g2, N_rf);

for idx_g = 1:N_g2
    temp_RAQR_config = RAQR_config; 
    temp_RAQR_config.gamma_2 = gamma_mat2(idx_g, 2); 
    temp_RAQR_config.gamma_3 = gamma_mat2(idx_g, 3); 
    temp_RAQR_config.gamma_4 = gamma_mat2(idx_g, 4); 

    for idx = 1:N_rf
        [~, probeResponseSteady] = getSteadySolution(temp_RAQR_config, RF_strengths(idx)); 
        pT2(idx_g, idx) = db2pow(probeResponseSteady); 
    end

end



%% Visualizations 
saveFigs = false; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',    12);
set(0,'DefaultAxesFontSize',    14);
set(0,'DefaultLineLineWidth',   1.2);
set(0,'defaultfigurecolor',     'w');

mycmap1 = othercolor('RdYlBu_11b', N_g1); 


fig1 = figure(1); 
cell_legends = cell(1, N_g1);

yyaxis left; 
for idx_g = 1:N_g1
    plot(RF_strengths, pT1(idx_g, :), 'color', mycmap1(idx_g, :), 'Marker', 'none', 'LineStyle', '-'); hold on; 
    cell_legends{idx_g} = sprintf("$\\gamma_4=%.2f\\,{\\rm kHz}$", gamma_mat1(idx_g, 4)/1e3/(2*pi)); 
end

set(gca, 'xscale', 'log'); 
xlabel('$E_{\rm LO}\,{\rm (V/m)}$', 'Interpreter', 'latex'); 
ylabel('DC Probe transmission'); 

yyaxis right; 
for idx_g = 1:N_g1
    plot(RF_strengths(2:end), diff(pT1(idx_g, :)), 'color', mycmap1(idx_g, :), 'LineStyle', '-.', 'marker', 'none'); hold on; 
    cell_legends{N_g1+idx_g} = ""; 
end
legend(cell_legends, 'interpreter', 'latex'); 
ylabel('Derivative'); 


fig2 = figure(2); 

mycmap2 = othercolor('RdYlBu_11b', N_g2); 
cell_legends = cell(1, N_g2);

yyaxis left; 
for idx_g = 1:N_g2
    plot(RF_strengths, pT2(idx_g, :), 'color', mycmap2(idx_g, :), 'Marker', 'none', 'LineStyle', '-'); hold on; 
    cell_legends{idx_g} = sprintf("$\\gamma_3=%.2f\\,{\\rm kHz}$", gamma_mat2(idx_g, 3)/1e3/(2*pi)); 
end
set(gca, 'xscale', 'log'); 
xlabel('$E_{\rm LO}\,{\rm (V/m)}$', 'Interpreter', 'latex'); 
ylabel('DC Probe transmission'); 

yyaxis right; 
for idx_g = 1:N_g2
    plot(RF_strengths(2:end), diff(pT2(idx_g, :)), 'color', mycmap2(idx_g, :), 'LineStyle', '-.', 'marker', 'none'); hold on; 
    cell_legends{N_g2+idx_g} = ""; 
end

legend(cell_legends, 'interpreter', 'latex'); 
ylabel('Derivative'); 
% title('\gamma_3 and \gamma_4 adjusted'); 



