clear; clc; close all;

epsilon_0   = 8.85e-12; 
mu_0        = 4*pi*1e-7; 
hbar        = 6.626e-34 / (2*pi); 
c0          = physconst('LightSpeed'); 

f       = 6.9458e9;          
k0      = 2*pi*f/c0; 
lambda  = c0/f; 

d   = 2e-2;                 % 2cm
N0  = (4.89e10)*(1e6);      % number of atoms per m^3 

Delta_p = 2*pi*(0e6); 
Delta_c = 2*pi*0e6;
Delta_l = 2*pi*0e3; 

gamma   = 0;
gamma_2 = 2*pi*5.2e6; 
gamma_3 = 2*pi*3.9e3; 
gamma_4 = 2*pi*1.7e3; 
gamma_c = 0; 

a0      = 5.2918e-11; 
e       = 1.6e-19; 
% mu_MW   = 1200*e*a0; 
mu_MW   = 1443.45*e*a0; 
mu_12   = 4.5022*e*a0; 

Omega_c     = 2*pi*2.05e6;              % coupling beam 17.0mW  @ 510nm
Omega_p     = 2*pi*8.08e6;               % probe    beam 29.8uW  @ 852nm
lambda_c    = 510e-9; 
lambda_p    = 852e-9; 

E_RF        = 0.04;         
Omega_RF    = mu_MW*(E_RF)/hbar;        % 1V/m

S_RF        = E_RF^2/(2*120*pi); 
A_ant       = lambda^2/(4*pi); 
P_ant       = S_RF*A_ant; 
P_ant_dBm   = pow2db(P_ant*1000); 


% Quantum state definition 
%  |1> 6S1/2    |2> 6P3/2   |3> 47D5/2          |4> 48P3/2 
Delta_p = -Delta_p; Delta_c = -Delta_c; Delta_l = -Delta_l; 

H = [0,         Omega_p/2,  0,                  0;                  ...
    Omega_p/2,  Delta_p,    Omega_c/2,          0;                  ...
    0,          Omega_c/2,  Delta_p+Delta_c,    Omega_RF/2;         ...
    0,          0,          Omega_RF/2,         Delta_p+Delta_c+Delta_l]; 

Gamma = diag([gamma, gamma+gamma_2, gamma+gamma_3+gamma_c, gamma+gamma_4]); 


fc = 6.9458e9; 

% Re-order the elements of ρ to construct a vector differential equation 
% dx/dt = Ax + b

% Aorder = [  1, 1; ...
%             2, 2; ...
%             3, 3; ...
%             4, 4; ...
%             1, 2; ...
%             1, 3; ...
%             1, 4; ...
%             2, 3; ...
%             2, 4; ...
%             3, 4; ...
%             2, 1; ...
%             3, 1; ...
%             3, 2; ...
%             4, 1; ...
%             4, 2; ...
%             4, 3; ];  

Aorder = [  1, 1; ...
            2, 1; ...
            3, 1; ...
            4, 1; ...
            1, 2; ...
            2, 2; ...
            3, 2; ...
            4, 2; ...
            1, 3; ...
            2, 3; ...
            3, 3; ...
            4, 3; ...
            1, 4; ...
            2, 4; ...
            3, 4; ...
            4, 4; ];  

[Nvar, ~]   = size(Aorder); 
A           = zeros(Nvar); 
b           = zeros(Nvar, 1); 
b(1)        = gamma;    

for ii = 1:Nvar
    for jj = 1:Nvar
        m = Aorder(ii, 1); 
        n = Aorder(ii, 2); 
        k = Aorder(jj, 1); 
        l = Aorder(jj, 2); 
        A(ii, jj) = getA(m, n, k, l, H, Gamma, gamma, gamma_2, gamma_3, gamma_4); 
    end
end

% Try to find the steady-state solution for Ax+b=0. 
u4 = zeros(1, 16); u4([1, 6, 11, 16]) = 1; 

Aext        = [A; 1e4*u4];  
[U, S, V]   = svd(Aext); 
s           = diag(S); 
tmp         = (U'*[zeros(16, 1); 1e4]); 
xvec_steady = V*inv(diag(s))*tmp(1:16); 

fprintf('Steady-state rho2_1='); 
disp(xvec_steady(2)); 

% Perform symmetrization on xvec_steady
rho_steady = reshape(xvec_steady, [4,4]); 
rho_steady = (rho_steady+rho_steady')/2; 
xvec_steady = rho_steady(:); 

% You may find some more characteristics about the matrix A. 


[PA, DA]    = eig(A);      % Matrix A has one eigenvalue 0. This corresponds to the steady-state. 
lambdas     = diag(DA); 

% Impulse in the H(3,4) element 
A_ii = 10; 

A_another = -1i*(kron(eye(4), H) - kron(H.', eye(4))) - (1/2)*(kron(eye(4), Gamma) + kron(Gamma, eye(4))); 
A_another(1, 6) = A_another(1, 6) + gamma_2; 
A_another(1, 16) = A_another(1, 16) + gamma_4; 
A_another(6, 11) = A_another(6, 11) + gamma_3; 

% compare
% Check that they are equal 
% norm(A_another - A)

%% Now we want to determine the impulse response function of H34/H43 to rho2_1. 
% Indices of A is well-defined now 

Qtmp = eye(16); 
Qtmp(:,1) = (u4.')/norm(u4); 
[Q, R] = qr(Qtmp); 
Q(:,1) = -Q(:, 1); 

% (y = Q'*x) is the variable substituion 
A_new = Q'*A*Q; 
assert(norm(A_new(1, :)) < 1e-6); % The first row of A is all-zero. 

% Eqn: dy(2:n)/dt = y(1)*w + A_tilde*y(2:n), where y1(t) is always
% constant = (1/2).
w       = A_new(2:end, 1); 
A_tilde = A_new(2:end, 2:end); 

[Vtilde, Dtilde] = eig(A_tilde); 

% Let y(2:16) = z
z_steady = -(A_tilde)\((1/2)*w); 

% Recover x_steady from z_steady 
y_steady = [(1/2); z_steady]; 
x_steady = Q*y_steady; 


% Evaluate the characteristic polynomial of A_tilde 
pA_tilde = poly(A_tilde); 

syms s; 
tmp = s*eye(15)-A_tilde; 

pA2 = det(tmp);   
pB = cell(15);  
for ii = 1:15
    for jj = 1:15
        rsel = true([1, 15]); 
        csel = true([1, 15]);
        rsel(ii) = false; csel(jj) = false; 
        tmp2 = tmp(rsel, csel); 
        pB{jj, ii} = (-1)^(ii+jj)*simplify(det(tmp2)); 
    end
end

save('minors.mat', "A_tilde", "Q", "w", "x_steady", "pB", "pA2"); 
fprintf('Complete.\n'); 
% [vAr, dAr] = cdf2rdf(vA, dA); 
% It seems that A has four real-valued eigenvalues, among which one is 0.
% Other eigen-values appear in pairs. 

% The (3,4)-element in H appears many times in A. They are: 
Pos_ni = [3, 4;  7, 8;     11, 12;  15, 16]; 
Pos_pi = [9, 13; 10, 14;   11, 15;  12, 16]; 

C34   = zeros(16); 
for ii = 1:4
    k           = Pos_ni(ii, 1); 
    ell         = Pos_ni(ii, 2); 
    x_init      = zeros(16, 1); 
    x_init(k)   = xvec_steady(ell);
    x_init      = x_init - xvec_steady;

    [dC, p]     = getAnalyticalODESolution(A, x_init);  
    C34         = C34 + (-1i)*dC; 
end

for ii = 1:4
    k           = Pos_pi(ii, 1); 
    ell         = Pos_pi(ii, 2); 
    x_init      = zeros(16, 1); 
    x_init(k)   = xvec_steady(ell);
    x_init      = x_init - xvec_steady; 

    [dC, p]     = getAnalyticalODESolution(A, x_init);  
    C34         = C34 + (+1i)*dC; 
end

Pos_ni = Pos_ni(:, [2, 1]); 
Pos_pi = Pos_pi(:, [2, 1]); 
C43 = zeros(16); 
for ii = 1:4
    k           = Pos_ni(ii, 1); 
    ell         = Pos_ni(ii, 2); 
    x_init      = zeros(16, 1); 
    x_init(k)   = xvec_steady(ell);
    x_init      = x_init - xvec_steady;      

    [dC, p]     = getAnalyticalODESolution(A, x_init);  
    C43           = C43 + (-1i)*dC; 
end

for ii = 1:4
    k           = Pos_pi(ii, 1); 
    ell         = Pos_pi(ii, 2); 
    x_init      = zeros(16, 1); 
    x_init(k)   = xvec_steady(ell);
    x_init      = x_init - xvec_steady; 

    [dC, p]     = getAnalyticalODESolution(A, x_init);  
    C43           = C43 + (+1i)*dC; 
end

% Plot here for C34 and C43 impulse response 
dht     = 1e-9; 
hT      = 5e-6; 
ht_arr  = 0:dht:hT; 
hNt     = length(ht_arr); 


h34_arr = C34(2, :) * exp(p*ht_arr); 
h43_arr = C43(2, :) * exp(p*ht_arr); 
h_arr = h34_arr + h43_arr; 

figure(1); 
subplot(3, 1, 1); 
plot(ht_arr, real(h34_arr)); hold on; 
plot(ht_arr, imag(h34_arr)); 

subplot(3, 1, 2); 
plot(ht_arr, real(h43_arr)); hold on; 
plot(ht_arr, imag(h43_arr)); 

subplot(3, 1, 3); 
plot(ht_arr, real(h_arr)); hold on; 
plot(ht_arr, imag(h_arr)); 


figure(2);
f_arr   = logspace(1, 8, 5000); 
H34_arr = C34(2,:) * (1./(-1i*2*pi*f_arr - p)); 
H43_arr = C43(2,:) * (1./(1i*2*pi*f_arr - p)); 
H_arr = H34_arr+H43_arr; 

subplot(3, 1, 1); 
plot(f_arr, mag2db(abs(H34_arr))); hold on; 
xlabel('f (Hz)'); 
set(gca, 'xscale', 'log'); 


subplot(3, 1, 2); 
plot(f_arr, mag2db(abs(H43_arr))); hold on; 
xlabel('f (Hz)'); 
set(gca, 'xscale', 'log'); 

subplot(3, 1, 3); 
plot(f_arr, mag2db(abs(H_arr))); hold on; 
xlabel('f (Hz)'); 
set(gca, 'xscale', 'log'); 



%% Standard solution for solving dx/dt = Ax+b
% first we solve the case where b==0
dt      = 1e-9;
T       = 100e-6;
t_arr   = 0:dt:T; 
Nt      = length(t_arr);
initial_rho = [0.088, 0.024, 0.888, 0.000].'; 



x_arr       = zeros(Nvar, Nt); 
x_arr(1:4, 1) = initial_rho; 


for idx_t = 2:length(t_arr)
    t = t_arr(idx_t); 
    x_arr(:, idx_t) = expm(t*A)*x_arr(:,1); 
end

% Alternative method with Runge-Kutta formula 
rho_arr         = zeros(16, Nt); 
rho_0           = diag(initial_rho); 
rho_arr(:,1)    = rho_0(:); 

for idx_t = 1:(Nt-1)
    rho_i   = reshape(rho_arr(:, idx_t), [4, 4]); 
    Lambda  = diag([gamma+gamma_2*rho_i(2,2)+gamma_4*rho_i(4,4), gamma_3*rho_i(3,3), 0, 0]); 

    k1      = dt*((-1i)*(H*rho_i-rho_i*H) - (1/2)*(Gamma*rho_i+rho_i*Gamma) + Lambda); 
    rho_ik1 = rho_i + (1/2)*reshape(k1, [4, 4]); 

    k2      = dt*((-1i)*(H*rho_ik1-rho_ik1*H) - (1/2)*(Gamma*rho_ik1+rho_ik1*Gamma) + Lambda); 
    rho_ik2 = rho_i + (1/2)*reshape(k2, [4, 4]); 
    
    k3      = dt*((-1i)*(H*rho_ik2-rho_ik2*H) - (1/2)*(Gamma*rho_ik2+rho_ik2*Gamma) + Lambda); 
    rho_ik3 = rho_i + reshape(k3, [4, 4]); 
    
    k4      = dt*((-1i)*(H*rho_ik3-rho_ik3*H) - (1/2)*(Gamma*rho_ik3+rho_ik3*Gamma) + Lambda); 

    rho_arr(:, idx_t+1) = rho_arr(:, idx_t) + (k1(:)+2*k2(:)+2*k3(:)+k4(:))/6; 

end


%% Visualization 
close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');

fig1 = figure(1); 
fig1.Position = [100, 100, 600, 600]; 
mycmap = othercolor('RdYlBu_11b', 4); 

subplot(2, 1, 1); 
plot(t_arr*1e6, real(x_arr(1, :)), 'color', mycmap(1, :));  hold on; 
plot(t_arr*1e6, real(x_arr(2, :)), 'color', mycmap(2, :)); 
plot(t_arr*1e6, real(x_arr(3, :)), 'color', mycmap(3, :));  
plot(t_arr*1e6, real(x_arr(4, :)), 'color', mycmap(4, :));  

xlabel('time/us'); 
ylabel('Density matrix element');
legend({"\rho_1_1", "\rho_2_2", "\rho_3_3", "\rho_4_4"}); 
title('Analytical'); 

subplot(2, 1, 2); 
plot(t_arr*1e6, real(rho_arr(1, :)),   'color', mycmap(1, :));  hold on; 
plot(t_arr*1e6, real(rho_arr(6, :)),   'color', mycmap(2, :)); 
plot(t_arr*1e6, real(rho_arr(11, :)),  'color', mycmap(3, :));  
plot(t_arr*1e6, real(rho_arr(16, :)),  'color', mycmap(4, :));  

xlabel('time/us'); 
ylabel('Density matrix element');
legend({"\rho_1_1", "\rho_2_2", "\rho_3_3", "\rho_4_4"}); 
title('Runge-Kutta method'); 

fig2 = figure(2); 
fig2.Position = [920, 100, 600, 600];

subplot(2, 1, 1); 
plot(t_arr*1e6, real(x_arr(5, :))); hold on; 
plot(t_arr*1e6, imag(x_arr(5, :))); 
xlabel('time/us'); 
ylabel('\rho_1_2');
legend({'Re', 'Im'}); 

subplot(2, 1, 2);
lightAmpAttnExponent = (-pi*d/lambda_p)*imag(-(2*N0*mu_12^2)/(epsilon_0*hbar*Omega_p)*rho_arr(2,:)); 

plot(t_arr*1e6, real(rho_arr(2, :))); hold on; 
plot(t_arr*1e6, imag(rho_arr(2, :))); 
xlabel('time/us'); 
ylabel('\rho_2_1');
legend({'Re', 'Im'}); 

% figure(3); 
% plot(t_arr*1e6, abs(x_arr(1, :))+abs(x_arr(2, :))+abs(x_arr(3, :))+abs(x_arr(4, :)));



%% Function utils


function A = getA(m, n, k, l, H, Gamma, gamma, gamma_2, gamma_3, gamma_4)
    A = 0; 

    % -i (Hρ) term
    A = A + (-1i)*(n==l)*H(m,k); 
    
    % i(ρH) term
    A = A + (1i)*(m==k)*H(l,n); 

    % -(1/2)Gamma*ρ term
    A = A - (1/2)*(n==l)*Gamma(m,k); 

    % -(1/2)ρ*Gamma term
    A = A - (1/2)*(m==k)*Gamma(l,n); 

    if k == 2 && l == 2 && m == 1 && n == 1
        A = A + gamma_2;
    end

    if k == 4 && l == 4 && m == 1 && n == 1
        A = A + gamma_4; 
    end

    if k == 3 && l == 3 && m == 2 && n == 2
        A = A + gamma_3; 
    end

end


