function [rhoArr, probeResponse] = transientRAQRSimulation(RARQ_config, sim_config, inputEfield)
    % Physical constants
    epsilon_0   = 8.85e-12; 
    mu_0        = 4*pi*1e-7; 
    hbar        = 6.626e-34 / (2*pi); 
    c0          = physconst('LightSpeed'); 

    a0      = 5.2918e-11; 
    e       = 1.6e-19; 
    mu_MW   = 1443.45*e*a0; 
    mu_12   = 4.5022*e*a0; 

    d = RARQ_config.d; 
    N0 = RARQ_config.N0; 
    
    % Extract RARQ configurations
    Delta_p = RARQ_config.Delta_p; 
    Delta_c = RARQ_config.Delta_c; 
    Delta_l = RARQ_config.Delta_l; 
    
    
    gamma   = RARQ_config.gamma; 
    gamma_2 = RARQ_config.gamma_2; 
    gamma_3 = RARQ_config.gamma_3; 
    gamma_4 = RARQ_config.gamma_4; 
    gamma_c = RARQ_config.gamma_c; 
    
    
    
    Omega_c = RARQ_config.Omega_c; 
    Omega_p = RARQ_config.Omega_p; 
    lambda_c = RARQ_config.lambda_c; 
    lambda_p = RARQ_config.lambda_p; 
    
    % Compute Rabi frequencies for the RF input sequence 
    Omega_RFseq    = mu_MW*(inputEfield)/hbar;      

    % Set-up transient simulation parameters 
    dt = sim_config.dt; 
    Nt = sim_config.Nt; 

    
    
    
    % Solve the master equation with Runge-Kutta method 
    rho_arr         = zeros(16, Nt); 

    if isfield(sim_config, "initial_rho")
        rho_0 = sim_config.initial_rho; 

    else
        initial_rho = [0.088, 0.024, 0.888, 0.000].'; 
        rho_0           = diag(initial_rho); 

    end
    rho_arr(:,1)    = rho_0(:); 

    
    for idx_t = 1:(Nt-1)
        rho_i       = reshape(rho_arr(:, idx_t), [4, 4]); 
        Omega_RF    = Omega_RFseq(idx_t); 

        % Hamiltonian, relaxation an decay matrices 
        Lambda  = diag([gamma+gamma_2*rho_i(2,2)+gamma_4*rho_i(4,4), gamma_3*rho_i(3,3), 0, 0]); 

        
        %  |1> 6S1/2    |2> 6P3/2   |3> 47D5/2          |4> 48P3/2 
        H = [0,         Omega_p/2,  0,                      0;                  ...
            Omega_p/2,  -Delta_p,    Omega_c/2,             0;                  ...
            0,          Omega_c/2,  -Delta_p-Delta_c,       conj(Omega_RF)/2;   ...
            0,          0,          Omega_RF/2,            -Delta_p-Delta_c-Delta_l]; 
        
        Gamma = diag([gamma, gamma+gamma_2, gamma+gamma_3+gamma_c, gamma+gamma_4]); 

    
        k1      = dt*((-1i)*(H*rho_i-rho_i*H) - (1/2)*(Gamma*rho_i+rho_i*Gamma) + Lambda); 
        rho_ik1 = rho_i + (1/2)*reshape(k1, [4, 4]); 
    
        k2      = dt*((-1i)*(H*rho_ik1-rho_ik1*H) - (1/2)*(Gamma*rho_ik1+rho_ik1*Gamma) + Lambda); 
        rho_ik2 = rho_i + (1/2)*reshape(k2, [4, 4]); 
        
        k3      = dt*((-1i)*(H*rho_ik2-rho_ik2*H) - (1/2)*(Gamma*rho_ik2+rho_ik2*Gamma) + Lambda); 
        rho_ik3 = rho_i + reshape(k3, [4, 4]); 
        
        k4      = dt*((-1i)*(H*rho_ik3-rho_ik3*H) - (1/2)*(Gamma*rho_ik3+rho_ik3*Gamma) + Lambda); 
    
        rho_arr(:, idx_t+1) = rho_arr(:, idx_t) + (k1(:)+2*k2(:)+2*k3(:)+k4(:))/6; 
    
    end

    rhoArr          = rho_arr; 
    chi_e           = -2*(N0*mu_12^2)/(epsilon_0*hbar*Omega_p)*rho_arr(2,:); 
    probeResponse   = (20/log(10))*(-pi*d/lambda_p)*imag(-(2*N0*mu_12^2)/(epsilon_0*hbar*Omega_p)*rho_arr(2,:)); 
    

end
