% ======================================================= % 
% 
%  This file provides a transient simulation platform for RAQRs.  
%  
%  
% ======================================================= % 

clear; close all; clc; 

RAQR_config = configureRAQR(); 


% Communication link parameters
Ptx_dBm     = 20;
txAntGain   = 5; 
Distance    = 1000;

eta0        = getPhysicalConstant("VacuumWaveImpedance"); 
rxEfield    = sqrt((2*eta0)*db2pow(Ptx_dBm-30+txAntGain)/(4*pi*Distance^2)); 

% Quantum system parameters
EAmp_LO         = 0.04;                         % LO field peak value           (V/m) 
% EAmp_Sig        = 0.0001;                       % Signal field peak value Ep    (V/m) 
EAmp_Sig        = rxEfield; 
f_IF            = 150e3;                        % Intermediate frequency of 200kHz
Ts              = 10e-6;                        % Symbol period is 10us (100kHz symbol rate)
fprintf('RAQR params set with Baseband BW = %.2f kHz and IF freq = %.2f kHz\n', (1/Ts)/1e3, f_IF/1e3); 

addNoise = false; 

%% Solve quantum transfer functions 
% [tfI, tfQ] = getPerturbationTransferFunctions(RAQR_config, EAmp_LO); 
% save('tfs2.mat', 'tfI', 'tfQ', 'RAQR_config'); 


%% Simulation 

% load tfs2.mat; 
% % Pre-processing of the transfer functions 
% GI1 = tf(real(tfI.Numerator{1}), tfI.Denominator{1}); 
% GI2 = tf(imag(tfI.Numerator{1}), tfI.Denominator{1}); 
% GQ1 = tf(real(tfQ.Numerator{1}), tfQ.Denominator{1}); 
% GQ2 = tf(imag(tfQ.Numerator{1}), tfQ.Denominator{1}); 
% dcGainMat = [dcgain(GI1), dcgain(GQ1); dcgain(GI2), dcgain(GQ2)];
% fprintf('DC Gain matrix = \n'); 
% disp(dcGainMat); 

% Transient simulation parameters
sim_config.dt       = 1e-9;
sim_config.T        = 2e-3;                     % Quantum simulation time-scale 
sim_config.t_arr    = 0:sim_config.dt:sim_config.T-sim_config.dt; 
sim_config.Nt       = length(sim_config.t_arr);

% Burn-in period parameters
burnin.dt       = 1e-9;
burnin.T        = 250e-6;               
burnin.t_arr    = 0:burnin.dt:burnin.T-burnin.dt; 
burnin.Nt       = length(burnin.t_arr);

[rho_arr0, ~] = transientRAQRSimulation(RAQR_config, burnin, EAmp_LO*ones(1, burnin.Nt)); 
% The burn-in result can also be obtained by steady-state solution 
[rho_ss, ~] = getSteadySolution(RAQR_config, EAmp_LO); 


% Synthesize transmit waveform 
rng(1); 
% transmitSymbols = [zeros(1, 40), (1-2*( rand([1, 60])>0.5 )) + 1i*(1-2*( rand([1, 60])>0.5 ))]; 
Nsyms       = sim_config.T/Ts; 
ratio_zero  = 0.1; 
Nzsyms      = round(ratio_zero*Nsyms/2); 
Nisyms      = Nsyms - Nzsyms*2; 

transmitSymbols = [zeros(1, Nzsyms), qammod(randi(16, [1, Nisyms])-1, 16, 'UnitAveragePower', true), zeros(1, Nzsyms)]; 
[E_RF, X_sig]   = synthesizeEfield(sim_config, EAmp_LO, EAmp_Sig, f_IF, 1/Ts, transmitSymbols); 



% Classical receiver: Compute the dipole antenna received power. 
f       = 6.9458e9;   
lambda  = physconst('LightSpeed')/f; 
S_RF    = EAmp_Sig^2/(2*120*pi); 
A_ant   = lambda^2/(4*pi); 
P_ant   = S_RF*A_ant; 
P_ant_dBm = pow2db(P_ant*1000); 
fprintf('Dipole received power = %.2f dBm\n', P_ant_dBm); 

% Add BBR noise to the synthesized waveform
c0      = getPhysicalConstant("LightSpeed"); 
kB      = getPhysicalConstant("Boltzmann"); 
B_nu    = 2*(f/c0)^2*(kB*300);                                      % BBR spectral radiance 
sigmac  = sqrt((8*pi/3)*eta0*B_nu*(1/sim_config.dt/2));             % complex noise variance per sample, at sampling freq 1/(dt)
E_RF    = E_RF + addNoise*sigmac*(randn(size(E_RF)) + 1i*randn(size(E_RF)));  

% Transient RAQR response computation
sim_config.initial_rho      = reshape(rho_arr0(:, end), [4, 4]); 
[rho_arr, probeResponse]    = transientRAQRSimulation(RAQR_config, sim_config, E_RF); 

sel = (sim_config.t_arr > -1); 
fprintf('mean probe transmission = %.3f dB\n', mean(probeResponse(sel))); 
fprintf('P-P response = %.3f dB\n', max(probeResponse(sel))-min(probeResponse(sel))); 

% Theoretical RAQR response computation (without transient simulations and noises)
% a0          = 5.2918e-11; 
% 
% mu_MW       = 1443.45*e*a0; 
% hbar        = 6.626e-34 / (2*pi);
% Omega_Sig   = mu_MW*EAmp_Sig/hbar; 
% 
% resp_rho21  = (1i)*lsim(GI2, (Omega_Sig/2)*real(exp(1i*2*pi*f_IF*sim_config.t_arr) .* (X_sig)), sim_config.t_arr); 
% resp_rho21  = resp_rho21 + (1i)*lsim(GQ2, (Omega_Sig/2)*imag(exp(1i*2*pi*f_IF*sim_config.t_arr) .* (X_sig)), sim_config.t_arr); 
% 
% mu_12       = 4.5022*e*a0; 
% epsilon_0   = 8.85e-12;
% chi_e       = -2*(RAQR_config.N0*mu_12^2)/(epsilon_0*hbar*RAQR_config.Omega_p)*( resp_rho21.'+rho_ss(2,1) ); 
% theoreticalProbeResponse   = (20/log(10))*(-pi*RAQR_config.d/RAQR_config.lambda_p)*imag(chi_e); 


% ================ Use the quantum transconductance method to obtain
% theoretical probe response 

[gI1, gQ1, gI2, gQ2, success]   = findTransferFunctionFile(hashStruct(RAQR_config), './data/'); 

if ~success
    [gI1, gQ1, gI2, gQ2]    = getQuantumTransConductanceTFs(RAQR_config); % This function is the slowest. 
    save(fullfile(dataPath, 'tfs.mat'), 'hashString', 'gI1', 'gI2', 'gQ1', 'gQ2'); 

else
    % do nothing 
end
gq              = tf(gI2.b, gI2.a);  % do not save tf structure into .mat files.

resp_Isig       = lsim(gq, real(exp(1i*2*pi*f_IF*sim_config.t_arr) .* (X_sig*EAmp_Sig*RAQR_config.d) ), sim_config.t_arr); 
resp_Isig       = resp_Isig + lsim(tf(gQ2.b, gQ2.a), imag(exp(1i*2*pi*f_IF*sim_config.t_arr) .* (X_sig*EAmp_Sig*RAQR_config.d)), sim_config.t_arr); 

eta     = RAQR_config.eta; 
kp      = 2*pi/RAQR_config.lambda_p; 
hbar    = 6.626e-34 / (2*pi); 
e       = 1.6e-19; 
P_probe = 29.8e-6; 

Iph0    = getTransmittedProbePower0(RAQR_config); 
theoreticalProbeResponse = pow2db((resp_Isig+Iph0)*(hbar*c0*kp)/(e*eta*P_probe)); 

fprintf('Theoretical probe response generated.\n'); 


%% Receive opto-electric conversion 
Iph0    = (eta*P_probe)/(hbar*(c0*kp))*e*db2pow(probeResponse); 

RL          = 50;
T0          = 300; 
PSDIn_bb    = 2*kB*T0/RL;                      % Double-sided PSD in the output photocurrent.


% Trans-impedance amplifier PARAMETERS (TIA model: DHPCA-100 Variable Gain High-speed Current Amplifier)
TIA_RT      = 10e3;                     % Transimpedance gain                   [V/A] 
TIA_Ipsd    = (1.8e-12/sqrt(2))^2;      % TIA input referred current noise PSD  [A^2/Hz]
TIA_Vpsd    = (2.8e-9/sqrt(2))^2;       % TIA input referred voltage noise PSD  [V^2/Hz] 
TIA_Zin     = 60; 
TIA_Zout    = 50;       
PD_Rs       = 1000;                      % DC bias resistor of the PD 

Kc          = (PD_Rs)/(PD_Rs+TIA_Zin);
sigma_rV    = sqrt( (2*kB*T0/PD_Rs*Kc^2 + (TIA_Ipsd*Kc)^2 + (TIA_Vpsd/(TIA_Zin+PD_Rs))^2 ) * TIA_RT^2 / sim_config.dt);  
% Noise-adding formula: noise variance = (double-sided PSD) * (simulation sample rate)

% sigma_rI    = sqrt(PSDIn_bb/sim_config.dt);     % noise std, real, current (A) 

% AC response 
Vsig        = (Iph0-mean(Iph0))*TIA_RT;    % voltage signal after TIA 

if addNoise
    Vsig    = Vsig + sigma_rV*randn(size(Iph0)); 
    fprintf('Photocurrent noise added\n'); 
else
    fprintf('Photocurrent noise not added.\n');
end

%% Visualization 
close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');
saveFigs = false; 

% Plot the non-burn-in period response 
fig1 = figure(1); 
fig1.Position = [100, 100, 600, 800]; 

subplot(2, 1, 1); 
mycmap = othercolor('RdYlBu_11b', 4); 
plot(sim_config.t_arr*1e6, real(rho_arr(1, :)),   'color', mycmap(1, :));  hold on; 
plot(sim_config.t_arr*1e6, real(rho_arr(6, :)),   'color', mycmap(2, :)); 
plot(sim_config.t_arr*1e6, real(rho_arr(11, :)),  'color', mycmap(3, :));  
plot(sim_config.t_arr*1e6, real(rho_arr(16, :)),  'color', mycmap(4, :));  

xlabel('time (us)'); 
ylabel('Density matrix element');
legend({"\rho_1_1", "\rho_2_2", "\rho_3_3", "\rho_4_4"}); 
title('Diagonal elements');

subplot(2, 1, 2); 
mycmap = othercolor('RdYlBu_11b', 6); 
plot(sim_config.t_arr*1e6, real(rho_arr(2, :)),  'color', mycmap(1, :));  hold on; 
plot(sim_config.t_arr*1e6, real(rho_arr(3, :)),  'color', mycmap(2, :)); 
plot(sim_config.t_arr*1e6, real(rho_arr(4, :)),  'color', mycmap(3, :));  
plot(sim_config.t_arr*1e6, real(rho_arr(7, :)),  'color', mycmap(4, :));  
plot(sim_config.t_arr*1e6, real(rho_arr(8, :)),  'color', mycmap(5, :));  
plot(sim_config.t_arr*1e6, real(rho_arr(12, :)), 'Color', mycmap(6, :));  

xlabel('time (us)'); 
ylabel('Density matrix element');
legend({"Re\rho_2_1", "Re\rho_3_1", "Re\rho_4_1", "Re\rho_2_3", "Re\rho_2_4", "Re\rho_4_3"}); 
title('Non-diagonal elements');


% Plot the burn-in period response 
fig2 = figure(2); 
fig2.Position = [100, 100, 600, 800]; 

subplot(2, 1, 1); 
mycmap = othercolor('RdYlBu_11b', 4); 
plot(burnin.t_arr*1e6, real(rho_arr0(1, :)),   'color', mycmap(1, :));  hold on; 
plot(burnin.t_arr*1e6, real(rho_arr0(6, :)),   'color', mycmap(2, :)); 
plot(burnin.t_arr*1e6, real(rho_arr0(11, :)),  'color', mycmap(3, :));  
plot(burnin.t_arr*1e6, real(rho_arr0(16, :)),  'color', mycmap(4, :));  

xlabel('time (us)'); 
ylabel('Density matrix element');
legend({"\rho_1_1", "\rho_2_2", "\rho_3_3", "\rho_4_4"}); 
title('Diagonal elements');

subplot(2, 1, 2); 
mycmap = othercolor('RdYlBu_11b', 6); 
plot(burnin.t_arr*1e6, real(rho_arr0(2, :)),  'color', mycmap(1, :));  hold on; 
plot(burnin.t_arr*1e6, real(rho_arr0(3, :)),  'color', mycmap(2, :)); 
plot(burnin.t_arr*1e6, real(rho_arr0(4, :)),  'color', mycmap(3, :));  
plot(burnin.t_arr*1e6, real(rho_arr0(7, :)),  'color', mycmap(4, :));  
plot(burnin.t_arr*1e6, real(rho_arr0(8, :)),  'color', mycmap(5, :));  
plot(burnin.t_arr*1e6, real(rho_arr0(12, :)), 'Color', mycmap(6, :));  

xlabel('time (us)'); 
ylabel('Density matrix element');
legend({"Re\rho_2_1", "Re\rho_3_1", "Re\rho_4_1", "Re\rho_2_3", "Re\rho_2_4", "Re\rho_4_3"}); 
title('Non-diagonal elements');



% Show the communication results 
fig3 = figure(3); 
fig3.Position = [920, 100, 600, 700];

subplot(3, 1, 1); 
plot(sim_config.t_arr*1e6, EAmp_Sig*real(X_sig)); hold on; 
plot(sim_config.t_arr*1e6, EAmp_Sig*imag(X_sig));
xlabel('time (us)'); 
ylabel('Input RF field (V/m)');
legend({'I', 'Q'}); 

subplot(3, 1, 2); 
% plot(sim_config.t_arr*1e6, EAmp_Sig*real(rho_arr(2, :)), 'Color', 'r'); hold on;
plot(sim_config.t_arr*1e6, EAmp_Sig*imag(rho_arr(2, :)), 'Color', 'b');
xlabel('time (us)'); 
ylabel('\rho_2_1'); 
% legend({'Real', 'Imag'}); 
legend('Imag[\rho_2_1]'); 

subplot(3, 1, 3); 
plot(sim_config.t_arr*1e6, (probeResponse)); hold on; 
plot(sim_config.t_arr*1e6, theoreticalProbeResponse); 
xlabel('time (us)'); 
ylabel('Probe transmission (dB)');
legend({'Sim', 'Theory'}); 

if saveFigs
   exportgraphics(fig3, "results/waveform.pdf", "ContentType", "vector");  
end

% small time-scale waveform 
fig4 = figure(4); 
fig4.Position = [446,74,558,389];
tsel = and((sim_config.t_arr) < 300e-6, sim_config.t_arr >= 200e-6); 
plot(sim_config.t_arr(tsel)*1e6, probeResponse(tsel)); hold on; 
plot(sim_config.t_arr(tsel)*1e6, theoreticalProbeResponse(tsel)); 
xlabel('time (us)'); 
ylabel('Probe transmission (dB)');
legend({'Sim', 'Theory'}, 'Location', 'southwest'); 
if saveFigs
    exportgraphics(fig4, "results/waveform_fine.pdf", "ContentType", "vector");
end


%% Fig 5-6: Simulation results for wireless communications 
% tmp     = db2pow(probeResponse);
saveFigs    = false; 
tmp         = Vsig - mean(Vsig); 

IFSignal    = resample(tmp, 1, 1e3);                        % IF Signal sample Rate = 1MHz with IF 
tIF_arr     = sim_config.t_arr; 
tIF_arr     = tIF_arr(1:1e3:end); 

fig5 = figure(5); hold off; 
% fig5.Position = [446,74,1200,600];
fig5.Position = [446, 74, 560, 420]; 
subplot(2, 2, 1); 
plot(tIF_arr*1e6, IFSignal); 
xlabel('time (us)'); ylabel('Rx IF signal (V)'); 

subplot(2, 2, 2); 
plotFFTSpectrum(IFSignal, sim_config.dt*1e3); 
ylabel('Rx IF spectrum (dBV)'); 

% Down-conversion to baseband 
nCutoffFreq = 0.15; 
[b, a]      = butter(10, nCutoffFreq);                      % cutoff frequency (IF sample rate/2)*w
fprintf('IF Mixer LPF cutoff freq = %.2f kHz\n', (1/sim_config.dt/1e3/2)*nCutoffFreq/1e3); 
I_BB = filter(b, a, IFSignal.*cos(2*pi*f_IF*tIF_arr));      % IF-stage down-conversion IQ mixer 
Q_BB = filter(b, a, IFSignal.*(-sin(2*pi*f_IF*tIF_arr))); 

% Match filter 
dsrate      = round(Ts/(1e3*sim_config.dt)); 
sinc_mf     = sinc(-4:1/(dsrate):4); 
delay_mf    = floor(length(sinc_mf)/2); 
I_mf        = conv(I_BB, sinc_mf); 
Q_mf        = conv(Q_BB, sinc_mf); 

I_mf        = I_mf(delay_mf+1:end-delay_mf); 
Q_mf        = Q_mf(delay_mf+1:end-delay_mf); 
IQ_BB       = (I_BB+1i*Q_BB)*exp(1j*deg2rad(180)); 

subplot(2, 2, 3); 
plot(tIF_arr*1e6, I_BB); hold on; 
plot(tIF_arr*1e6, Q_BB);
legend({"I", "Q"}); 
xlabel('time (us)');
ylabel('Rx baseband signal (a.u.)'); 


subplot(2, 2, 4); 
plotFFTSpectrum(IQ_BB, sim_config.dt*1e3); 
ylabel('Signal spectrum (dB)'); 

if saveFigs
    exportgraphics(fig5, "results/spectra.pdf", "ContentType", "vector");
end


% Fig 6 contains the Tx/Rx baseband signals and the constellations. 
fig6 = figure(6); hold off; 
fig6.Position = [530,210,560,420];

subplot(2, 2, 1); 
plot(sim_config.t_arr*1e6, EAmp_Sig*real(X_sig)); hold on; 
plot(sim_config.t_arr*1e6, EAmp_Sig*imag(X_sig));
xlabel('time (us)'); 
ylabel('Input RF field (V/m)');
legend({'I', 'Q'}); 

subplot(2, 2, 2); 
txNonZeroSymbols = transmitSymbols(Nzsyms+1:end-Nzsyms); 
scatter(real(txNonZeroSymbols), imag(txNonZeroSymbols), 6*ones(size(txNonZeroSymbols)), 'Marker', 'o', 'MarkerFaceColor', 'flat'); 
xlabel('I'); ylabel('Q'); axis equal; box on; 
set(gca, 'xlim', [-1.2, 1.2]); 
set(gca, 'ylim', [-1.2, 1.2]); 

subplot(2, 2, 3); 
plot(tIF_arr*(1e6), real(IQ_BB)); hold on; 
plot(tIF_arr*(1e6), imag(IQ_BB)); 
xlabel('time (us)'); 
ylabel('Recv IQ signal');  
legend({'I', 'Q'}); 

subplot(2, 2, 4); 
sampleDelay = 14; 
fprintf('Delay correction = %.2f us\n', sampleDelay); 
recoverIQSymbols = [IQ_BB(1+sampleDelay:round(Ts/(1e3*sim_config.dt)):end), 0]; 
recoverIQSymbols = recoverIQSymbols(Nzsyms+1:Nsyms-Nzsyms); 
% recoverIQSymbols = resample(IQ_BB, 1, round(Ts/(1e3*sim_config.dt)) ); 
scatter(real(recoverIQSymbols), imag(recoverIQSymbols), 6*ones(size(recoverIQSymbols)), 'Marker', 'o', 'MarkerFaceColor','flat'); 
xlabel('I'); ylabel('Q'); axis equal; box on; 
set(gca, 'xlim', [-6.1, 6.1]*1e-4); 
set(gca, 'ylim', [-6.1, 6.1]*1e-4); 

h_est       = mean(recoverIQSymbols./transmitSymbols(Nzsyms+1:Nsyms-Nzsyms));
nvar_est    = mean(abs(recoverIQSymbols-h_est*transmitSymbols(Nzsyms+1:Nsyms-Nzsyms)).^2); 
snr_est     = abs((h_est))^2/nvar_est; 

fprintf('Estimated SNR = %.2f dB\n', pow2db(snr_est)); 

if saveFigs
    exportgraphics(fig6, "results/constellations.pdf", "ContentType", "vector");
    fprintf('Files exported\n'); 
end

% Fig.7 monitors the transmitted spectrum 
fig7 = figure(7); hold off; 
plotFFTSpectrum(X_sig, sim_config.dt); 

fprintf('Figures generated.\n'); 

%% Theory-predicted SNR 






%% Utils

function [totalEfield, Xsig] = synthesizeEfield(sim_config, A_LO, A_Sig, f_IF, bbSampleRate, bbIQsamples)
% Input sequences are in row vectors.
% Output: totalEfield (V/m), sigEfield(dimensionless) 
% TODO: thermal noise field is also added here. 
    dt      = sim_config.dt; 
    t_arr   = sim_config.t_arr; 
    Nt      = sim_config.Nt; 
    N_IQ    = length(bbIQsamples); 
    
    totalEfield     = A_LO*ones([1, Nt]); 
    upsampleRate    = ceil((1/dt)/(bbSampleRate)); 

    if A_Sig > 0
        % sampleTimes         = (0:N_IQ-1)/(bbSampleRate); 
        % resampleIQsamples   = interp1(sampleTimes, bbIQsamples, t_arr, "cubic"); 
        % upsampleFilter = ones([1, upsampleRate]);           % design the smoothing filter

        Nsinc           = 6; 
        upsampleFilter  = sinc((-Nsinc:1/upsampleRate:Nsinc)); 
        % upsampleFilter = ones([1, upsampleRate]);

        % sigEfield   = resample(bbIQsamples, upsampleRate, 1); 
        tmp             = upsample(bbIQsamples, upsampleRate); % impulse train of spacing upsampleRate

        tmp             = conv(tmp, upsampleFilter); 
        lfilter         = floor(length(upsampleFilter)/2); 

        if mod(length(upsampleFilter), 2) == 0
            Xsig   = tmp(lfilter:end-lfilter); 
        else
            Xsig   = tmp(lfilter+1:end-lfilter); 
        end
        
        % Output field (complex envelope) = LO + Sig
        totalEfield     = totalEfield + A_Sig*exp(1i*2*pi*f_IF*t_arr).*Xsig; 

    end

end

function plotFFTSpectrum(x, ts)
    Nx  = length(x); 
    X   = fftshift(fft(x)); 
    fs  = -(1/ts/2):(1/(Nx*ts)):(1/ts/2-1); 
    
    plot(fs/1e3, mag2db(abs(X)));
    xlabel('Freq (kHz)'); 
end

