% ======================================================= % 
% 
%  This file provides a single-carrier simulation platform for RAQRs.  
%  
%  
% ======================================================= % 

clear; close all; clc; 

RAQR_config = configureRAQR(); 
addNoise    = true; 

% Communication link parameters
Ptx_dBm     = 10;
txAntGain   = 5; 
Distance    = 2e3;
eta0        = getPhysicalConstant("VacuumWaveImpedance"); 
rxEfield    = sqrt((2*eta0)*db2pow(Ptx_dBm-30+txAntGain)/(4*pi*Distance^2)); 

% Quantum system parameters
EAmp_LO         = 0.04;                         % LO field peak value           (V/m) 
% EAmp_Sig        = 0.0001;                       % Signal field peak value Ep    (V/m) 
EAmp_Sig        = rxEfield; 
f_IF            = 150e3;                        % Intermediate frequency of 200kHz
Ts              = 10e-6;                        % Symbol period is 10us (100kHz symbol rate)
fprintf('RAQR params set with Baseband BW = %.2f kHz and IF freq = %.2f kHz\n', (1/Ts)/1e3, f_IF/1e3); 


%% Simulation 

% Transient simulation parameters
sim_config.dt       = 1e-9;
sim_config.T        = 10e-3;                     % Quantum simulation time-scale 
sim_config.t_arr    = 0:sim_config.dt:sim_config.T-sim_config.dt; 
sim_config.Nt       = length(sim_config.t_arr);
sim_config.Fs_IF    = 2e6; 
sim_config.Nrcl     = 20; 
sim_config.beta     = 0.4; 

% Burn-in period parameters
burnin.dt       = 1e-9;
burnin.T        = 300e-6;               
burnin.t_arr    = 0:burnin.dt:burnin.T-burnin.dt; 
burnin.Nt       = length(burnin.t_arr);

[rho_arr0, ~] = transientRAQRSimulation(RAQR_config, burnin, EAmp_LO*ones(1, burnin.Nt)); 
% The burn-in result can also be obtained by steady-state solution 
[rho_ss, ~] = getSteadySolution(RAQR_config, EAmp_LO); 


% Synthesize transmit waveform 
rng(1); 
% transmitSymbols = [zeros(1, 40), (1-2*( rand([1, 60])>0.5 )) + 1i*(1-2*( rand([1, 60])>0.5 ))]; 
Nsyms       = round(sim_config.T/Ts); 
ratio_zero  = 0.1; 
Nzsyms      = round(ratio_zero*Nsyms/2); 
Nisyms      = Nsyms - Nzsyms*2; 
ModOrder    = 2^4; 

symSel          = randi(ModOrder, [1, Nisyms]); 
transmitSymbols = [zeros(1, Nzsyms), qammod(symSel-1, ModOrder, 'UnitAveragePower', true), zeros(1, Nzsyms)]; 
% transmitSymbols = [zeros(1, Nzsyms), ones(1, Nisyms), zeros(1, Nzsyms)]; 
[E_RF, X_sig]   = waveformMod(sim_config, EAmp_LO, EAmp_Sig, f_IF, 1/Ts, transmitSymbols, "rcosine_sqrt"); 


% Classical receiver: Compute the dipole antenna received power. 
f       = 6.9458e9;   
lambda  = physconst('LightSpeed')/f; 
S_RF    = EAmp_Sig^2/(2*120*pi); 
A_ant   = lambda^2/(4*pi); 
P_ant   = S_RF*A_ant; 
P_ant_dBm = pow2db(P_ant*1000); 
T0      = 300; 

fprintf('Dipole received power = %.2f dBm\n', P_ant_dBm); 

% Add BBR noise to the synthesized waveform
c0      = getPhysicalConstant("LightSpeed"); 
kB      = getPhysicalConstant("Boltzmann"); 
B_nu    = 2*(f/c0)^2*(kB*T0);                   % BBR spectral radiance 
sigma2c = (8*pi/3)*eta0*B_nu*(1/sim_config.dt); % complex noise variance per sample, at sampling freq 1/(dt). See plan.md for details. 
E_RF    = E_RF + addNoise*sqrt(sigma2c)*(randn(size(E_RF)) + 1i*randn(size(E_RF)))/sqrt(2); 


% Transient RAQR response computation
sim_config.initial_rho      = reshape(rho_arr0(:, end), [4, 4]); 
[rho_arr, probeResponse]    = transientRAQRSimulation(RAQR_config, sim_config, E_RF); 

sel = (sim_config.t_arr > -1); 
fprintf('mean probe transmission = %.3f dB\n', mean(probeResponse(sel))); 
fprintf('P-P response = %.3f dB\n', max(probeResponse(sel))-min(probeResponse(sel))); 

% ================ Use the quantum transconductance method to obtain theoretical probe response ================

[gI1, gQ1, gI2, gQ2, success]   = findTransferFunctionFile(hashStruct(RAQR_config), './data/'); 

if ~success
    [gI1, gQ1, gI2, gQ2]    = getQuantumTransConductanceTFs(RAQR_config); % This function is the slowest. 
    save(fullfile(dataPath, 'tfs.mat'), 'hashString', 'gI1', 'gI2', 'gQ1', 'gQ2'); 

else
    % do nothing 
end
gq              = tf(gI2.b, gI2.a);  % do not save tf structure into .mat files.

resp_Isig       = lsim(gq, real(exp(1i*2*pi*f_IF*sim_config.t_arr) .* (X_sig*EAmp_Sig*RAQR_config.d) ), sim_config.t_arr); 
resp_Isig       = resp_Isig + lsim(tf(gQ2.b, gQ2.a), imag(exp(1i*2*pi*f_IF*sim_config.t_arr) .* (X_sig*EAmp_Sig*RAQR_config.d)), sim_config.t_arr); 

eta     = RAQR_config.eta; 
kp      = 2*pi/RAQR_config.lambda_p; 
hbar    = 6.626e-34 / (2*pi); 
e       = 1.6e-19; 
P_probe = 29.8e-6; 

Iph0    = getTransmittedProbePower0(RAQR_config); 
theoreticalProbeResponse = pow2db((resp_Isig+Iph0)*(hbar*c0*kp)/(e*eta*P_probe)); 

fprintf('Theoretical probe response generated.\n'); 


%% Receive opto-electric conversion 
Iph0    = (eta*P_probe)/(hbar*(c0*kp))*e*db2pow(probeResponse); 

RL          = 50;
T0          = 300; 
PSDIn_bb    = 2*kB*T0/RL;                      % Double-sided PSD in the output photocurrent.


% Trans-impedance amplifier PARAMETERS (TIA model: DHPCA-100 Variable Gain High-speed Current Amplifier)
TIA_RT      = 10e3;                     % Transimpedance gain                   [V/A] 
TIA_Ipsd    = (1.8e-12/sqrt(2))^2;      % TIA input referred current noise PSD  [A^2/Hz]
TIA_Vpsd    = (2.8e-9/sqrt(2))^2;       % TIA input referred voltage noise PSD  [V^2/Hz] 
TIA_Zin     = 60; 
TIA_Zout    = 50;       
PD_Rs       = 1000;                      % DC bias resistor of the PD 

Kc          = (PD_Rs)/(PD_Rs+TIA_Zin);

% Noise-adding formula: noise variance = (double-sided PSD) * (simulation sample rate)
sigma_rV    = sqrt( (2*kB*T0/PD_Rs*Kc^2 + (TIA_Ipsd*Kc)^2 + (TIA_Vpsd/(TIA_Zin+PD_Rs))^2 ) * TIA_RT^2 / sim_config.dt);  


% AC response 
Vsig        = (Iph0-mean(Iph0))*TIA_RT;    % voltage signal after TIA 

if addNoise
    Vsig    = Vsig + sigma_rV*randn(size(Iph0)); 
    fprintf('Photocurrent noise added\n'); 
else
    fprintf('Photocurrent noise not added.\n');
end


%% Visualization 
close all; 
set(0,'DefaultLineMarkerSize',  6);
set(0,'DefaultTextFontSize',   12);
set(0,'DefaultAxesFontSize',   14);
set(0,'DefaultLineLineWidth',  1.5);
set(0,'defaultfigurecolor','w');
saveFigs = false; 

% Plot the non-burn-in period response 
fig1 = figure(1); 
fig1.Position = [100, 100, 600, 800]; 

subplot(2, 1, 1); 
mycmap = othercolor('RdYlBu_11b', 4); 
plot(sim_config.t_arr*1e6, real(rho_arr(1, :)),   'color', mycmap(1, :));  hold on; 
plot(sim_config.t_arr*1e6, real(rho_arr(6, :)),   'color', mycmap(2, :)); 
plot(sim_config.t_arr*1e6, real(rho_arr(11, :)),  'color', mycmap(3, :));  
plot(sim_config.t_arr*1e6, real(rho_arr(16, :)),  'color', mycmap(4, :));  

xlabel('time (us)'); 
ylabel('Density matrix element');
legend({"\rho_1_1", "\rho_2_2", "\rho_3_3", "\rho_4_4"}); 
title('Diagonal elements');

subplot(2, 1, 2); 
mycmap = othercolor('RdYlBu_11b', 6); 
plot(sim_config.t_arr*1e6, real(rho_arr(2, :)),  'color', mycmap(1, :));  hold on; 
plot(sim_config.t_arr*1e6, real(rho_arr(3, :)),  'color', mycmap(2, :)); 
plot(sim_config.t_arr*1e6, real(rho_arr(4, :)),  'color', mycmap(3, :));  
plot(sim_config.t_arr*1e6, real(rho_arr(7, :)),  'color', mycmap(4, :));  
plot(sim_config.t_arr*1e6, real(rho_arr(8, :)),  'color', mycmap(5, :));  
plot(sim_config.t_arr*1e6, real(rho_arr(12, :)), 'Color', mycmap(6, :));  

xlabel('time (us)'); 
ylabel('Density matrix element');
legend({"Re\rho_2_1", "Re\rho_3_1", "Re\rho_4_1", "Re\rho_2_3", "Re\rho_2_4", "Re\rho_4_3"}); 
title('Non-diagonal elements');


% Plot the burn-in period response 
fig2 = figure(2); 
fig2.Position = [100, 100, 600, 800]; 

subplot(2, 1, 1); 
mycmap = othercolor('RdYlBu_11b', 4); 
plot(burnin.t_arr*1e6, real(rho_arr0(1, :)),   'color', mycmap(1, :));  hold on; 
plot(burnin.t_arr*1e6, real(rho_arr0(6, :)),   'color', mycmap(2, :)); 
plot(burnin.t_arr*1e6, real(rho_arr0(11, :)),  'color', mycmap(3, :));  
plot(burnin.t_arr*1e6, real(rho_arr0(16, :)),  'color', mycmap(4, :));  

xlabel('time (us)'); 
ylabel('Density matrix element');
legend({"\rho_1_1", "\rho_2_2", "\rho_3_3", "\rho_4_4"}); 
title('Diagonal elements');

subplot(2, 1, 2); 
mycmap = othercolor('RdYlBu_11b', 6); 
plot(burnin.t_arr*1e6, real(rho_arr0(2, :)),  'color', mycmap(1, :));  hold on; 
plot(burnin.t_arr*1e6, real(rho_arr0(3, :)),  'color', mycmap(2, :)); 
plot(burnin.t_arr*1e6, real(rho_arr0(4, :)),  'color', mycmap(3, :));  
plot(burnin.t_arr*1e6, real(rho_arr0(7, :)),  'color', mycmap(4, :));  
plot(burnin.t_arr*1e6, real(rho_arr0(8, :)),  'color', mycmap(5, :));  
plot(burnin.t_arr*1e6, real(rho_arr0(12, :)), 'Color', mycmap(6, :));  

xlabel('time (us)'); 
ylabel('Density matrix element');
legend({"Re\rho_2_1", "Re\rho_3_1", "Re\rho_4_1", "Re\rho_2_3", "Re\rho_2_4", "Re\rho_4_3"}); 
title('Non-diagonal elements');



% Show the communication results 
fig3 = figure(3); 
fig3.Position = [920, 100, 600, 700];

subplot(3, 1, 1); 
plot(sim_config.t_arr*1e6, EAmp_Sig*real(X_sig)); hold on; 
plot(sim_config.t_arr*1e6, EAmp_Sig*imag(X_sig));
xlabel('time (us)'); 
ylabel('Input RF field (V/m)');
legend({'I', 'Q'}); 

subplot(3, 1, 2); 
% plot(sim_config.t_arr*1e6, EAmp_Sig*real(rho_arr(2, :)), 'Color', 'r'); hold on;
plot(sim_config.t_arr*1e6, EAmp_Sig*imag(rho_arr(2, :)), 'Color', 'b');
xlabel('time (us)'); 
ylabel('\rho_2_1'); 
% legend({'Real', 'Imag'}); 
legend('Imag[\rho_2_1]'); 

subplot(3, 1, 3); 
plot(sim_config.t_arr*1e6, (probeResponse)); hold on; 
plot(sim_config.t_arr*1e6, theoreticalProbeResponse.'); 
xlabel('time (us)'); 
ylabel('Probe transmission (dB)');
legend({'Sim', 'Theory'}); 

if saveFigs
   exportgraphics(fig3, "results/waveform.pdf", "ContentType", "vector");  
end

% small time-scale waveform 
fig4 = figure(4); 
fig4.Position = [446,74,558,389];
tsel = and((sim_config.t_arr) < 700e-6, sim_config.t_arr >= 600e-6); 
plot(sim_config.t_arr(tsel)*1e6, probeResponse(tsel)); hold on; 
plot(sim_config.t_arr(tsel)*1e6, theoreticalProbeResponse(tsel).'); 
xlabel('time (us)'); 
ylabel('Probe transmission (dB)');
legend({'Sim', 'Theory'}, 'Location', 'southwest'); 
if saveFigs
    exportgraphics(fig4, "results/waveform_fine.pdf", "ContentType", "vector");
end


%% Fig 5-6: Simulation results for wireless communications 

L           = round(sim_config.Fs_IF/(1/Ts)); 
L1          = round(1/sim_config.dt/sim_config.Fs_IF);      % down-sample rate from RK-RF to IF

saveFigs    = false; 
tmp         = Vsig - mean(Vsig); 

IFSignal    = resample(tmp, 1, L1);                        % IF Signal sample Rate = 1MHz with IF 
tIF_arr     = sim_config.t_arr; 
tIF_arr     = tIF_arr(1:L1:end); 

fig5 = figure(5); hold off; 
% fig5.Position = [446,74,1200,600];
fig5.Position = [364,89,1217,813]; 
subplot(3, 2, 1); 
plot(tIF_arr*1e6, IFSignal); 
xlabel('time (us)'); ylabel('Rx IF signal (V)'); 

subplot(3, 2, 2); 
plotFFTSpectrum(IFSignal, sim_config.dt*L1); 
ylabel('Rx IF spectrum (dBV)'); 

% Down-conversion to baseband 
nCutoffFreq     = 0.10; 
[b1, a1]        = butter(10, nCutoffFreq);                      % cutoff frequency (IF sample rate/2)*w
fprintf('IF Mixer LPF cutoff freq = %.2f kHz\n', (sim_config.Fs_IF/2)*nCutoffFreq/1e3); 
I_BB    = filter(b1, a1, IFSignal.*cos(2*pi*f_IF*tIF_arr));     % IF-stage down-conversion IQ mixer 
Q_BB    = filter(b1, a1, IFSignal.*(-sin(2*pi*f_IF*tIF_arr))); 
IQ_BB   = I_BB + 1i*Q_BB; 

% Match filter 
b2          = rcosdesign(sim_config.beta, sim_config.Nrcl, L, "sqrt")/sqrt(L);  % assume unit DC gain 

delay_mf    = (length(b2)-1)/2;  % # delay of samples  
I_mf        = conv(I_BB, b2); 
Q_mf        = conv(Q_BB, b2); 

I_mf        = I_mf(delay_mf+1:end-delay_mf); 
Q_mf        = Q_mf(delay_mf+1:end-delay_mf); 
IQ_BB2      = (I_mf+1i*Q_mf)*exp(-1j*deg2rad(133.45));  

subplot(3, 2, 3); 
plot(tIF_arr*1e6, I_BB); hold on; 
plot(tIF_arr*1e6, Q_BB);
legend({"I", "Q"}); 
xlabel('time (us)');
ylabel('BB filtered signal (a.u.)'); 

subplot(3, 2, 4); 
plotFFTSpectrum(IQ_BB, sim_config.dt*L1); 
ylabel('BB filtered spectrum (dB)'); 

subplot(3, 2, 5); 
plot(tIF_arr*1e6, real(IQ_BB2)); hold on; 
plot(tIF_arr*1e6, imag(IQ_BB2));
legend({"I", "Q"}); 
xlabel('time (us)');
ylabel('Sqrt RCos MF (a.u.)'); 

subplot(3, 2, 6); 
plotFFTSpectrum(IQ_BB2, sim_config.dt*L1); 
ylabel('Sqrt RCos MF (dB)'); 

if saveFigs
    exportgraphics(fig5, "results/spectra.pdf", "ContentType", "vector");
end


% Fig 6 contains the Tx/Rx baseband signals and the constellations. 
fig6 = figure(6); hold off; 
fig6.Position = [530, 210, 560, 420];

subplot(2, 2, 1); 
plot(sim_config.t_arr*1e6, EAmp_Sig*real(X_sig)); hold on; 
plot(sim_config.t_arr*1e6, EAmp_Sig*imag(X_sig));
xlabel('time (us)'); 
ylabel('Input RF field (V/m)');
legend({'I', 'Q'}); 

subplot(2, 2, 2); 
txNonZeroSymbols = transmitSymbols(Nzsyms+1:end-Nzsyms); 
scatter(real(txNonZeroSymbols), imag(txNonZeroSymbols), 6*ones(size(txNonZeroSymbols)), 'Marker', 'o', 'MarkerFaceColor', 'flat'); 
xlabel('I'); ylabel('Q'); axis equal; box on; 
set(gca, 'xlim', [-1.2, 1.2]); 
set(gca, 'ylim', [-1.2, 1.2]); 

subplot(2, 2, 3); 
plot(tIF_arr*(1e6), real(IQ_BB2)); hold on; 
plot(tIF_arr*(1e6), imag(IQ_BB2)); 
xlabel('time (us)'); 
ylabel('Recv IQ signal');  
legend({'I', 'Q'}); 

subplot(2, 2, 4); 
% sampleDelay = 14; 
sampleDelay = 22; 

fprintf('Delay correction = %.2f us\n', sampleDelay); 
recoverIQSymbols = [IQ_BB2(1+sampleDelay:round(Ts/(L1*sim_config.dt)):end), 0]; 
recoverIQSymbols = recoverIQSymbols(Nzsyms+1:Nsyms-Nzsyms); 

% scatter(real(recoverIQSymbols), imag(recoverIQSymbols), 6*ones(size(recoverIQSymbols)), 'Marker', 'o', 'MarkerFaceColor','flat'); 
% xlabel('I'); ylabel('Q'); axis equal; box on; 
showConstellation(recoverIQSymbols, symSel, ModOrder); 
% set(gca, 'xlim', [-2, 2]*1e-4); 
% set(gca, 'ylim', [-2, 2]*1e-4); 

h_est       = mean(recoverIQSymbols./transmitSymbols(Nzsyms+1:Nsyms-Nzsyms));
nvar_est    = mean(abs(recoverIQSymbols-h_est*transmitSymbols(Nzsyms+1:Nsyms-Nzsyms)).^2); 
snr_est     = abs((h_est))^2/nvar_est; 

fprintf('Estimated SNR = %.2f dB, h angle = %.2f deg, Capacity = %.2f bps/Hz\n', pow2db(snr_est), rad2deg(angle(h_est)), ...
    log2(1+snr_est)); 

if saveFigs
    exportgraphics(fig6, "results/constellations.pdf", "ContentType", "vector");
    fprintf('Files exported\n'); 
end

fig7 = figure(7); % eye graph 


fprintf('Figures generated.\n'); 

for idx = Nzsyms+1:Nsyms-Nzsyms
    Iseq = real(IQ_BB2((idx-1)*L+sampleDelay+1-L/2 : idx*L+sampleDelay-L/2 )); 
    Qseq = imag(IQ_BB2((idx-1)*L+sampleDelay+1-L/2 : idx*L+sampleDelay-L/2 ));

    hold on; 
    hPlot3 = plot3(-L/2:(L/2-1), Iseq, Qseq, 'Color', [0, 0, 0, 0.1]); hold on; 
end
xlabel('IF Sample'); 
ylabel('I'); 
zlabel('Q'); 

% fig8 = figure(8); 
% freqz(b1, a1); 
% ylabel('Butter response'); 


fprintf('End of this script.\n'); 

%% Utils
function [E_tot, X_sig] = waveformMod(sim_config, A_LO, A_sig, f_IF, F_sym, symSeq, method)
    Fs_IF   = sim_config.Fs_IF;   % IF simulation sample frequency 
    N_sym   = length(symSeq); 
    L       = round(Fs_IF / F_sym); 
    N_IF    = N_sym*L; 

    
    if method == "rcosine_sqrt"
        Nrcl    = sim_config.Nrcl;          % raised cosine filter length (in #symbols) 
        beta    = sim_config.beta;          % Exceed-bandwidth ratio
        b       = rcosdesign(beta, Nrcl, L, "sqrt");        % This filter does not change the sig energy in the digital domain. 
        Nfilter = length(b); 
        
        X_BB    = conv(b, sqrt(L)*upsample(symSeq, L));     % Baseband signal sampled by Fs_IF
        delay   = (Nfilter-1)/2; 
        X_BB    = X_BB(1+delay:N_IF+delay); 

    else
        error('%s not implemented.', method);
    end
    
    X_IF    = X_BB.*exp(1i*2*pi*(f_IF/Fs_IF)*(0:N_IF-1)); 
    Fs_rk   = 1/sim_config.dt;
    
    X_sig   = resample(X_BB, round(Fs_rk/Fs_IF), 1); 
    E_tot   = A_LO + A_sig*resample(X_IF, round(Fs_rk/Fs_IF), 1); 
    
end

function plotFFTSpectrum(x, ts)
    Nx  = length(x); 
    X   = fftshift(fft(x)); 
    fs  = -(1/ts/2):(1/(Nx*ts)):(1/ts/2-1); 
    
    plot(fs/1e3, mag2db(abs(X)));
    xlabel('Freq (kHz)'); 
end

function showConstellation(symSeq, symSel, ModOrder)
    colors = othercolor('RdBu10', ModOrder); 

    for idx = 1:ModOrder
        selSyms = symSeq(symSel == idx); 
        scatter(real(selSyms), imag(selSyms), 6*ones(size(selSyms)), colors(idx, :), 'Marker', 'o', 'MarkerFaceColor', 'flat'); hold on; 
    end
    
    xlabel('I'); ylabel('Q'); axis equal; box on; 

    % set(gca, 'xlim', [-1.2, 1.2]); 
    % set(gca, 'ylim', [-1.2, 1.2]); 

end

